"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGetLogsRateTimeseries = createGetLogsRateTimeseries;
var _utils = require("../../utils");
var _es_queries = require("../../utils/es_queries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createGetLogsRateTimeseries() {
  return async ({
    esClient,
    identifyingMetadata,
    serviceNames,
    timeFrom,
    timeTo,
    kuery,
    serviceEnvironmentQuery = []
  }) => {
    const intervalString = (0, _utils.getBucketSizeFromTimeRangeAndBucketCount)(timeFrom, timeTo, 50);

    // Note: Please keep the formula in `metricsFormulasMap` up to date with the query!

    const esResponse = await esClient.search({
      index: 'logs-*-*',
      size: 0,
      query: {
        bool: {
          filter: [...(0, _es_queries.kqlQuery)(kuery), {
            terms: {
              [identifyingMetadata]: serviceNames
            }
          }, ...serviceEnvironmentQuery, {
            range: {
              ['@timestamp']: {
                gte: timeFrom,
                lte: timeTo,
                format: 'epoch_millis'
              }
            }
          }]
        }
      },
      aggs: {
        service: {
          terms: {
            field: identifyingMetadata
          },
          aggs: {
            timeseries: {
              date_histogram: {
                field: '@timestamp',
                fixed_interval: `${intervalString}s`,
                min_doc_count: 0,
                extended_bounds: {
                  min: timeFrom,
                  max: timeTo
                }
              }
            }
          }
        }
      }
    });
    const aggregations = esResponse.aggregations;
    const buckets = aggregations === null || aggregations === void 0 ? void 0 : aggregations.service.buckets;
    return buckets ? buckets.reduce((acc, bucket) => {
      const timeseries = bucket.timeseries.buckets.map(timeseriesBucket => {
        return {
          x: timeseriesBucket.key,
          y: timeseriesBucket.doc_count / (intervalString / 60)
        };
      });
      return {
        ...acc,
        [bucket.key]: timeseries
      };
    }, {}) : {};
  };
}