"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deserializeVerboseTestOutput = exports.deserialize = void 0;
var _uuid = require("uuid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getProcessorType = processor => {
  /**
   * See the definition of {@link ProcessorInternal} for why this works to extract the
   * processor type.
   */
  const type = Object.keys(processor)[0];
  if (typeof type !== 'string') {
    throw new Error(`Invalid processor type. Received "${type}"`);
  }
  return type;
};
const convertToPipelineInternalProcessor = processor => {
  var _processor$type;
  const type = getProcessorType(processor);
  const {
    on_failure: originalOnFailure,
    ...options
  } = (_processor$type = processor[type]) !== null && _processor$type !== void 0 ? _processor$type : {};
  const onFailure = originalOnFailure !== null && originalOnFailure !== void 0 && originalOnFailure.length ? convertProcessors(originalOnFailure) : originalOnFailure;
  return {
    id: (0, _uuid.v4)(),
    type,
    onFailure,
    options
  };
};
const convertProcessors = processors => {
  const convertedProcessors = [];
  for (const processor of processors) {
    convertedProcessors.push(convertToPipelineInternalProcessor(processor));
  }
  return convertedProcessors;
};
const deserialize = ({
  processors,
  onFailure
}) => {
  return {
    processors: convertProcessors(processors),
    onFailure: onFailure ? convertProcessors(onFailure) : undefined
  };
};
exports.deserialize = deserialize;
/**
 * Find the previous state of the sample document in the pipeline
 * This typically will be the result from the previous processor
 * unless the previous processor had a "skipped" status
 */
const getProcessorInput = (processorIndex, document, count = 1) => {
  const previousProcessorIndex = processorIndex - count;
  if (previousProcessorIndex >= 0) {
    const processorResult = document.processor_results[previousProcessorIndex];
    if (!processorResult.doc) {
      const newCount = count + 1;
      return getProcessorInput(processorIndex, document, newCount);
    }
    return processorResult.doc;
  }
  return undefined;
};

/**
 * This function takes the verbose response of the simulate API
 * and maps the results to each processor in the pipeline by the "tag" field
 */
const deserializeVerboseTestOutput = output => {
  const {
    docs
  } = output;
  const deserializedOutput = docs.map(doc => {
    return doc.processor_results.reduce((processorResultsById, currentResult, index) => {
      const result = {
        ...currentResult
      };
      const resultId = result.tag;

      // We skip index 0, as the first processor will not have a previous result
      if (index !== 0) {
        result.processorInput = getProcessorInput(index, doc);
      }

      // The tag is added programatically as a way to map
      // the results to each processor
      // It is not something we need to surface to the user, so we delete it
      // @ts-expect-error
      delete result.tag;
      processorResultsById[resultId] = result;
      return processorResultsById;
    }, {});
  });
  return deserializedOutput;
};
exports.deserializeVerboseTestOutput = deserializeVerboseTestOutput;