"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.relationsSerializer = exports.relationsDeserializer = exports.RelationsParameter = void 0;
var _react = _interopRequireDefault(require("react"));
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _shared_imports = require("../../../shared_imports");
var _documentation = require("../../../../../services/documentation");
var _edit_field = require("../fields/edit_field");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// This is the Elasticsearch interface to declare relations

// Internally we will use this type for "relations" as it is more UI friendly
// to loop over the relations and its children

/**
 * Export custom serializer to be used when we need to serialize the form data to be sent to ES
 * @param field The field to be serialized
 */
const relationsSerializer = field => {
  if (field.relations === undefined) {
    return field;
  }
  const relations = field.relations;
  const relationsSerialized = relations.reduce((acc, item) => ({
    ...acc,
    [item.parent]: item.children.length === 1 ? item.children[0] : item.children
  }), {});
  return {
    ...field,
    relations: relationsSerialized
  };
};

/**
 * Export custom deserializer to be used when we need to deserialize the data coming from ES
 * @param field The field to be serialized
 */
exports.relationsSerializer = relationsSerializer;
const relationsDeserializer = field => {
  if (field.relations === undefined) {
    return field;
  }
  const relations = field.relations;
  const relationsDeserialized = Object.entries(relations).map(([parent, children]) => ({
    parent,
    children: typeof children === 'string' ? [children] : children
  }));
  return {
    ...field,
    relations: relationsDeserialized
  };
};
exports.relationsDeserializer = relationsDeserializer;
const childConfig = {
  defaultValue: []
};
const RelationsParameter = () => {
  const renderWarning = () => /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
    color: "warning",
    iconType: "warning",
    size: "s",
    title: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.idxMgmt.mappingsEditor.join.multiLevelsParentJoinWarningTitle",
      defaultMessage: "Avoid using multiple levels to replicate a relational model. Each relation level increases computation time and memory consumption at query time. For best performance, {docsLink}",
      values: {
        docsLink: /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
          href: _documentation.documentationService.getJoinMultiLevelsPerformanceLink(),
          target: "_blank"
        }, _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.join.multiLevelsPerformanceDocumentationLink', {
          defaultMessage: 'denormalize your data.'
        }))
      }
    })
  });
  return /*#__PURE__*/_react.default.createElement(_edit_field.EditFieldFormRow, {
    title: _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.relationshipsTitle', {
      defaultMessage: 'Relationships'
    }),
    withToggle: false
  }, /*#__PURE__*/_react.default.createElement(_shared_imports.UseArray, {
    path: "relations",
    initialNumberOfItems: 0
  }, ({
    items,
    addItem,
    removeItem
  }) => {
    const columns = [
    // Parent column
    {
      name: _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.relationshipTable.parentColumnTitle', {
        defaultMessage: 'Parent'
      }),
      render: item => {
        // By adding ".parent" to the path, we are saying that we want an **object**
        // to be created for each array item.
        // This object will have a "parent" property with the field value.
        return /*#__PURE__*/_react.default.createElement("div", {
          style: {
            width: '100%'
          }
        }, /*#__PURE__*/_react.default.createElement(_shared_imports.UseField, {
          path: `${item.path}.parent`,
          component: _shared_imports.TextField,
          componentProps: {
            euiFieldProps: {
              'aria-label': _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.relationshipTable.parentFieldAriaLabel', {
                defaultMessage: 'Parent field'
              })
            }
          }
          // For a newly created relation, we don't want to read
          // its default value provided to the form because... it is new! :)
          ,
          readDefaultValueOnForm: !item.isNew
        }));
      }
    },
    // Children column (ComboBox)
    {
      name: _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.relationshipTable.childrenColumnTitle', {
        defaultMessage: 'Children'
      }),
      render: item => {
        return /*#__PURE__*/_react.default.createElement("div", {
          style: {
            width: '100%'
          }
        }, /*#__PURE__*/_react.default.createElement(_shared_imports.UseField, {
          path: `${item.path}.children`,
          config: childConfig,
          component: _shared_imports.ComboBoxField,
          componentProps: {
            euiFieldProps: {
              'aria-label': _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.relationshipTable.childrenFieldAriaLabel', {
                defaultMessage: 'Children field'
              })
            }
          },
          readDefaultValueOnForm: !item.isNew
        }));
      }
    },
    // Actions column
    {
      width: '48px',
      actions: [{
        render: ({
          id
        }) => {
          const label = _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.relationshipTable.removeRelationshipTooltipLabel', {
            defaultMessage: 'Remove relationship'
          });
          return /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
            content: label,
            delay: "long",
            disableScreenReaderOutput: true
          }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
            "data-test-subj": "removeRelationshipButton",
            "aria-label": label,
            iconType: "minusInCircle",
            color: "danger",
            onClick: () => removeItem(id)
          }));
        }
      }]
    }];
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, items.length > 1 && /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, renderWarning(), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, null)), /*#__PURE__*/_react.default.createElement(_eui.EuiBasicTable, {
      items: items,
      itemId: "id",
      columns: columns,
      noItemsMessage: _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.relationshipTable.emptyTableMessage', {
        defaultMessage: 'No relationship defined'
      })
    }), /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
      onClick: addItem,
      iconType: "plusInCircleFilled",
      "data-test-subj": "addRelationButton"
    }, _i18n.i18n.translate('xpack.idxMgmt.mappingsEditor.joinType.addRelationshipButtonLabel', {
      defaultMessage: 'Add relationship'
    })));
  }));
};
exports.RelationsParameter = RelationsParameter;