"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createUninstallRoute = createUninstallRoute;
var _configSchema = require("@kbn/config-schema");
var _ebtTools = require("@kbn/ebt-tools");
var _utils = require("./utils");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createUninstallRoute(router, sampleDatasets, logger, usageTracker, analytics) {
  router.delete({
    path: '/api/sample_data/{id}',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because the uninstaller route is a wrapper around the Saved Object client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const routeStartTime = performance.now();
    const sampleDataset = sampleDatasets.find(({
      id
    }) => id === request.params.id);
    if (!sampleDataset) {
      return response.notFound();
    }
    const sampleDataInstaller = await (0, _utils.getSampleDataInstaller)({
      datasetId: sampleDataset.id,
      sampleDatasets,
      logger,
      context
    });
    try {
      await sampleDataInstaller.uninstall(request.params.id);
      // track the usage operation in a non-blocking way
      usageTracker.addUninstall(request.params.id);
      (0, _ebtTools.reportPerformanceMetricEvent)(analytics, {
        eventName: _utils.SAMPLE_DATA_UNINSTALLED_EVENT,
        duration: performance.now() - routeStartTime,
        key1: sampleDataset.id
      });
      return response.noContent();
    } catch (e) {
      if (e instanceof _errors.SampleDataInstallError) {
        return response.customError({
          body: {
            message: e.message
          },
          statusCode: e.httpCode
        });
      }
      throw e;
    }
  });
}