"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TYPE = exports.SyncIntegrationsTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _task = require("@kbn/task-manager-plugin/server/task");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../../common/constants");
var _services = require("../../services");
var _get = require("../../services/epm/packages/get");
var _fleet_synced_integrations = require("../../services/setup/fleet_synced_integrations");
var _sync_integrations_on_remote = require("./sync_integrations_on_remote");
var _custom_assets = require("./custom_assets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'fleet:sync-integrations-task';
const VERSION = exports.VERSION = '1.0.5';
const TITLE = 'Fleet Sync Integrations Task';
const SCOPE = ['fleet'];
const DEFAULT_INTERVAL = '5m';
const TIMEOUT = '5m';
class SyncIntegrationsTask {
  constructor(setupContract) {
    var _config$taskInterval;
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "taskInterval", void 0);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error('[SyncIntegrationsTask] Missing required service during start');
        return;
      }
      this.wasStarted = true;
      this.logger.info(`[SyncIntegrationsTask] Started with interval of [${this.taskInterval}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: TYPE,
          scope: SCOPE,
          schedule: {
            interval: this.taskInterval
          },
          state: {},
          params: {
            version: VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task SyncIntegrationsTask, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core) => {
      if (!this.wasStarted) {
        this.logger.debug('[SyncIntegrationsTask] runTask Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        this.logger.debug(`[SyncIntegrationsTask] Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      this.logger.info(`[runTask()] started`);
      if (!(0, _fleet_synced_integrations.canEnableSyncIntegrations)()) {
        this.logger.debug(`[SyncIntegrationsTask] Remote synced integration cannot be enabled.`);
        return;
      }
      const [coreStart, _startDeps, {
        packageService
      }] = await core.getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      try {
        // write integrations on main cluster
        await this.updateSyncedIntegrationsData(esClient, soClient);

        // sync integrations on remote cluster
        await (0, _sync_integrations_on_remote.syncIntegrationsOnRemote)(esClient, soClient, packageService.asInternalUser, this.abortController, this.logger);
        this.endRun('success');
      } catch (err) {
        if (err instanceof _elasticsearch.errors.RequestAbortedError) {
          this.logger.warn(`[SyncIntegrationsTask] request aborted due to timeout: ${err}`);
          this.endRun();
          return;
        }
        this.logger.error(`[SyncIntegrationsTask] error: ${err}`);
        this.endRun('error');
      }
    });
    (0, _defineProperty2.default)(this, "getSyncedIntegrationDoc", async esClient => {
      try {
        const res = await esClient.get({
          id: _fleet_synced_integrations.FLEET_SYNCED_INTEGRATIONS_INDEX_NAME,
          index: _fleet_synced_integrations.FLEET_SYNCED_INTEGRATIONS_INDEX_NAME
        });
        return res._source;
      } catch (error) {
        if (error.statusCode === 404) {
          return undefined;
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "hadAnyRemoteESSyncEnabled", remoteEsHosts => {
      return remoteEsHosts.some(host => host.sync_integrations);
    });
    (0, _defineProperty2.default)(this, "updateSyncedIntegrationsData", async (esClient, soClient) => {
      const outputs = await _services.outputService.list(soClient);
      const remoteESOutputs = outputs.items.filter(output => output.type === _constants.outputType.RemoteElasticsearch);
      const isSyncEnabled = remoteESOutputs.some(output => output.sync_integrations);
      if (isSyncEnabled) {
        await (0, _fleet_synced_integrations.createOrUpdateFleetSyncedIntegrationsIndex)(esClient);
      }
      const previousSyncIntegrationsData = await this.getSyncedIntegrationDoc(esClient);
      if (!isSyncEnabled) {
        const hadAnyRemoteESSyncEnabled = previousSyncIntegrationsData && this.hadAnyRemoteESSyncEnabled(previousSyncIntegrationsData.remote_es_hosts);
        if (!hadAnyRemoteESSyncEnabled) {
          return;
        }
      }
      const newDoc = {
        remote_es_hosts: remoteESOutputs.map(output => {
          var _remoteOutput$hosts, _remoteOutput$sync_in, _remoteOutput$sync_un;
          const remoteOutput = output;
          return {
            name: remoteOutput.name,
            hosts: (_remoteOutput$hosts = remoteOutput.hosts) !== null && _remoteOutput$hosts !== void 0 ? _remoteOutput$hosts : [],
            sync_integrations: (_remoteOutput$sync_in = remoteOutput.sync_integrations) !== null && _remoteOutput$sync_in !== void 0 ? _remoteOutput$sync_in : false,
            sync_uninstalled_integrations: (_remoteOutput$sync_un = remoteOutput.sync_uninstalled_integrations) !== null && _remoteOutput$sync_un !== void 0 ? _remoteOutput$sync_un : false
          };
        }),
        integrations: [],
        custom_assets: {}
      };
      const packageSavedObjects = await (0, _get.getInstalledPackageSavedObjects)(soClient, {
        perPage: _constants.SO_SEARCH_LIMIT,
        sortOrder: 'asc'
      });
      newDoc.integrations = packageSavedObjects.saved_objects.map(item => {
        var _item$updated_at;
        return {
          package_name: item.attributes.name,
          package_version: item.attributes.version,
          updated_at: (_item$updated_at = item.updated_at) !== null && _item$updated_at !== void 0 ? _item$updated_at : new Date().toISOString(),
          install_status: item.attributes.install_status,
          install_source: item.attributes.install_source
        };
      });
      const isSyncUninstalledEnabled = remoteESOutputs.some(output => output.sync_uninstalled_integrations);
      if (isSyncUninstalledEnabled && previousSyncIntegrationsData) {
        const removedIntegrations = previousSyncIntegrationsData.integrations.filter(item => !packageSavedObjects.saved_objects.map(data => data.attributes.name).includes(item.package_name));
        newDoc.integrations.push(...removedIntegrations.map(item => ({
          ...item,
          install_status: 'not_installed'
        })));
      }
      try {
        const customAssets = await (0, _custom_assets.getCustomAssets)(esClient, soClient, newDoc.integrations, this.abortController, previousSyncIntegrationsData);
        newDoc.custom_assets = (0, _lodash.keyBy)(customAssets, asset => `${asset.type}:${asset.name}`);
      } catch (error) {
        this.logger.warn(`[SyncIntegrationsTask] error getting custom assets: ${error}`);
        newDoc.custom_assets_error = {
          timestamp: new Date().toISOString(),
          error: error.message
        };
      }
      await esClient.index({
        id: _fleet_synced_integrations.FLEET_SYNCED_INTEGRATIONS_INDEX_NAME,
        index: _fleet_synced_integrations.FLEET_SYNCED_INTEGRATIONS_INDEX_NAME,
        body: newDoc
      }, {
        signal: this.abortController.signal
      });
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory,
      config
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    this.taskInterval = (_config$taskInterval = config.taskInterval) !== null && _config$taskInterval !== void 0 ? _config$taskInterval : DEFAULT_INTERVAL;
    _taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, _core);
            },
            cancel: async () => {
              this.abortController.abort('Task cancelled');
            }
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  endRun(msg = '') {
    this.logger.info(`[SyncIntegrationsTask] runTask ended${msg ? ': ' + msg : ''}`);
  }
}
exports.SyncIntegrationsTask = SyncIntegrationsTask;