"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.comparePreconfiguredPolicyToCurrent = comparePreconfiguredPolicyToCurrent;
exports.ensurePreconfiguredPackagesAndPolicies = ensurePreconfiguredPackagesAndPolicies;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _constants = require("../../common/constants");
var _constants2 = require("../constants");
var _simplified_package_policy_helper = require("../../common/services/simplified_package_policy_helper");
var _errors = require("../errors");
var _saved_object = require("./saved_object");
var _registry = require("./epm/registry");
var _packages = require("./epm/packages");
var _install = require("./epm/packages/install");
var _bulk_install_packages = require("./epm/packages/bulk_install_packages");
var _agent_policy = require("./agent_policy");
var _package_policy = require("./package_policy");
var _app_context = require("./app_context");
var _package_policy_frozen_variables = require("./preconfiguration/package_policy_frozen_variables");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function ensurePreconfiguredPackagesAndPolicies(defaultSoClient, esClient, policies = [], packages = [], defaultOutput, defaultDownloadSource, spaceId) {
  const logger = _app_context.appContextService.getLogger();

  // Validate configured packages to ensure there are no version conflicts
  const packageNames = (0, _lodash.groupBy)(packages, pkg => pkg.name);
  const duplicatePackages = Object.entries(packageNames).filter(([, versions]) => versions.length > 1);
  if (duplicatePackages.length) {
    // List duplicate packages as a comma-separated list of <package-name>:<semver>
    // If there are multiple packages with duplicate versions, separate them with semicolons, e.g
    // package-a:1.0.0, package-a:2.0.0; package-b:1.0.0, package-b:2.0.0
    const duplicateList = duplicatePackages.map(([, versions]) => versions.map(v => (0, _registry.pkgToPkgKey)(v)).join(', ')).join('; ');
    throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.preconfiguration.duplicatePackageError', {
      defaultMessage: 'Duplicate packages specified in configuration: {duplicateList}',
      values: {
        duplicateList
      }
    }));
  }
  const packagesToInstall = packages.map(pkg => pkg.version === _constants.PRECONFIGURATION_LATEST_KEYWORD ? {
    name: pkg.name,
    prerelease: pkg.prerelease,
    skipDataStreamRollover: pkg.skipDataStreamRollover
  } : pkg);

  // Preinstall packages specified in Kibana config
  const preconfiguredPackages = await (0, _bulk_install_packages.bulkInstallPackages)({
    savedObjectsClient: defaultSoClient,
    esClient,
    packagesToInstall,
    force: true,
    // Always force outdated packages to be installed if a later version isn't installed
    skipIfInstalled: true,
    // force flag alone would reinstall packages that are already installed
    spaceId
  });
  const fulfilledPackages = [];
  const rejectedPackages = [];
  for (let i = 0; i < preconfiguredPackages.length; i++) {
    const packageResult = preconfiguredPackages[i];
    if ('error' in packageResult) {
      logger.warn(`Failed installing package [${packages[i].name}] due to error: [${packageResult.error}]`);
      rejectedPackages.push({
        package: {
          name: packages[i].name,
          version: packages[i].version
        },
        error: packageResult.error
      });
    } else {
      fulfilledPackages.push(packageResult);
    }
  }

  // Keeping this outside of the Promise.all because it introduces a race condition.
  // If one of the required packages fails to install/upgrade it might get stuck in the installing state.
  // On the next call to the /setup API, if there is a upgrade available for one of the required packages a race condition
  // will occur between upgrading the package and reinstalling the previously failed package.
  // By moving this outside of the Promise.all, the upgrade will occur first, and then we'll attempt to reinstall any
  // packages that are stuck in the installing state.
  await (0, _install.ensurePackagesCompletedInstall)(defaultSoClient, esClient);

  // Create policies specified in Kibana config
  logger.debug(`Creating preconfigured policies`);
  const preconfiguredPolicies = await Promise.allSettled(policies.map(async preconfiguredAgentPolicy => {
    if (preconfiguredAgentPolicy.id) {
      // Check to see if a preconfigured policy with the same preconfiguration id was already deleted by the user
      const preconfigurationId = preconfiguredAgentPolicy.id.toString();
      const searchParams = {
        searchFields: ['id'],
        search: (0, _saved_object.escapeSearchQueryPhrase)(preconfigurationId)
      };
      const deletionRecords = await defaultSoClient.find({
        type: _constants2.PRECONFIGURATION_DELETION_RECORD_SAVED_OBJECT_TYPE,
        ...searchParams
      });
      const wasDeleted = deletionRecords.total > 0;
      if (wasDeleted) {
        return {
          created: false,
          deleted: preconfigurationId
        };
      }
    } else if (!preconfiguredAgentPolicy.is_default && !preconfiguredAgentPolicy.is_default_fleet_server) {
      throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.preconfiguration.missingIDError', {
        defaultMessage: '{agentPolicyName} is missing an `id` field. `id` is required, except for policies marked is_default or is_default_fleet_server.',
        values: {
          agentPolicyName: preconfiguredAgentPolicy.name
        }
      }));
    }
    const namespacedSoClient = preconfiguredAgentPolicy.space_id ? _app_context.appContextService.getInternalUserSOClientForSpaceId(preconfiguredAgentPolicy.space_id) : _app_context.appContextService.getInternalUserSOClientForSpaceId(_coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING);
    const {
      created,
      policy
    } = await _agent_policy.agentPolicyService.ensurePreconfiguredAgentPolicy(namespacedSoClient, esClient, (0, _lodash.omit)(preconfiguredAgentPolicy, 'is_managed', 'space_id') // Don't add `is_managed` until the policy has been fully configured and not persist space_id
    );
    if (!created) {
      if (!policy) return {
        created,
        policy,
        namespacedSoClient
      };
      if (!policy.is_managed && !preconfiguredAgentPolicy.is_managed) return {
        created,
        policy
      };
      const {
        hasChanged,
        fields
      } = comparePreconfiguredPolicyToCurrent(preconfiguredAgentPolicy, policy);
      const newFields = {
        download_source_id: defaultDownloadSource.id,
        ...fields
      };
      if (hasChanged) {
        const updatedPolicy = await _agent_policy.agentPolicyService.update(namespacedSoClient, esClient, String(preconfiguredAgentPolicy.id), newFields, {
          force: true
        });
        return {
          created,
          policy: updatedPolicy,
          namespacedSoClient
        };
      }
      return {
        created,
        policy,
        namespacedSoClient
      };
    }
    return {
      created,
      policy,
      namespacedSoClient,
      shouldAddIsManagedFlag: preconfiguredAgentPolicy.is_managed
    };
  }));
  const fulfilledPolicies = [];
  const rejectedPolicies = [];
  for (let i = 0; i < preconfiguredPolicies.length; i++) {
    const policyResult = preconfiguredPolicies[i];
    if (policyResult.status === 'rejected') {
      rejectedPolicies.push({
        error: policyResult.reason,
        agentPolicy: {
          name: policies[i].name
        }
      });
      continue;
    }
    fulfilledPolicies.push(policyResult.value);
    const {
      created,
      policy,
      shouldAddIsManagedFlag,
      namespacedSoClient
    } = policyResult.value;
    if (created || policies[i].is_managed) {
      if (!namespacedSoClient) {
        throw new Error('No soClient created for that policy');
      }
      const preconfiguredAgentPolicy = policies[i];
      const {
        package_policies: packagePolicies
      } = preconfiguredAgentPolicy;
      const agentPolicyWithPackagePolicies = await _agent_policy.agentPolicyService.get(namespacedSoClient, policy.id, true);
      const installedPackagePolicies = await Promise.all(packagePolicies.map(async preconfiguredPackagePolicy => {
        const {
          package: pkg,
          ...newPackagePolicy
        } = preconfiguredPackagePolicy;
        const installedPackage = await (0, _packages.getInstallation)({
          savedObjectsClient: defaultSoClient,
          pkgName: pkg.name
        });
        if (!installedPackage) {
          const rejectedPackage = rejectedPackages.find(rp => {
            var _rp$package;
            return ((_rp$package = rp.package) === null || _rp$package === void 0 ? void 0 : _rp$package.name) === pkg.name;
          });
          if (rejectedPackage) {
            throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.preconfiguration.packageRejectedError', {
              defaultMessage: `[{agentPolicyName}] could not be added. [{pkgName}] could not be installed due to error: [{errorMessage}]`,
              values: {
                agentPolicyName: preconfiguredAgentPolicy.name,
                pkgName: pkg.name,
                errorMessage: rejectedPackage.error.toString()
              }
            }));
          }
          throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.preconfiguration.packageMissingError', {
            defaultMessage: '[{agentPolicyName}] could not be added. [{pkgName}] is not installed, add [{pkgName}] to [{packagesConfigValue}] or remove it from [{packagePolicyName}].',
            values: {
              agentPolicyName: preconfiguredAgentPolicy.name,
              packagePolicyName: newPackagePolicy.name,
              pkgName: pkg.name,
              packagesConfigValue: 'xpack.fleet.packages'
            }
          }));
        }
        return {
          installedPackage,
          packagePolicy: newPackagePolicy,
          namespacedSoClient
        };
      }));
      const [packagePoliciesToAdd, packagePoliciesUpdates] = installedPackagePolicies.reduce((acc, installablePackagePolicy) => {
        var _agentPolicyWithPacka;
        const isAdd = !(agentPolicyWithPackagePolicies === null || agentPolicyWithPackagePolicies === void 0 ? void 0 : agentPolicyWithPackagePolicies.package_policies).some(packagePolicy => packagePolicy.id !== undefined && packagePolicy.id === installablePackagePolicy.packagePolicy.id || packagePolicy.name === installablePackagePolicy.packagePolicy.name);
        if (isAdd) {
          acc[0].push(installablePackagePolicy);
          return acc;
        }
        if (!agentPolicyWithPackagePolicies) {
          return acc;
        }
        const existingPackagePolicy = (_agentPolicyWithPacka = agentPolicyWithPackagePolicies.package_policies) === null || _agentPolicyWithPacka === void 0 ? void 0 : _agentPolicyWithPacka.find(packagePolicy => packagePolicy.id === installablePackagePolicy.packagePolicy.id || packagePolicy.name === installablePackagePolicy.packagePolicy.name);
        if (!existingPackagePolicy) {
          return acc;
        }
        const updatePackagePolicy = {
          ...existingPackagePolicy
        };
        if (Array.isArray(installablePackagePolicy.packagePolicy.inputs)) {
          if ((0, _package_policy_frozen_variables.packagePolicyHasFrozenVariablesUpdate)(updatePackagePolicy, installablePackagePolicy.packagePolicy.inputs)) {
            acc[1].push({
              namespacedSoClient: installablePackagePolicy.namespacedSoClient,
              packagePolicy: updatePackagePolicy,
              inputs: installablePackagePolicy.packagePolicy.inputs
            });
          }
        }
        return acc;
      }, [[], []]);
      logger.debug(() => `Adding preconfigured package policies ${JSON.stringify(packagePoliciesToAdd.map(pol => ({
        name: pol.packagePolicy.name,
        package: pol.installedPackage.name
      })))}`);
      const s = _elasticApmNode.default.startSpan('Add preconfigured package policies', 'preconfiguration');
      await addPreconfiguredPolicyPackages(esClient, policy, packagePoliciesToAdd, defaultOutput);
      s === null || s === void 0 ? void 0 : s.end();
      logger.debug(() => `Updating preconfigured package policies ${JSON.stringify(packagePoliciesUpdates.map(pol => {
        var _pol$packagePolicy$pa;
        return {
          name: pol.packagePolicy.name,
          package: (_pol$packagePolicy$pa = pol.packagePolicy.package) === null || _pol$packagePolicy$pa === void 0 ? void 0 : _pol$packagePolicy$pa.name
        };
      }))}`);
      const s2 = _elasticApmNode.default.startSpan('Update preconfigured package policies', 'preconfiguration');
      for (const packagePolicyUpdate of packagePoliciesUpdates) {
        try {
          await (0, _package_policy_frozen_variables.updateFrozenInputs)(esClient, packagePolicyUpdate.namespacedSoClient, packagePolicyUpdate.packagePolicy, packagePolicyUpdate.inputs);
        } catch (error) {
          logger.error(`Error updating preconfigured variables for package policy ${packagePolicyUpdate.packagePolicy.name}: ${error.message}`, {
            error
          });
        }
      }
      s2 === null || s2 === void 0 ? void 0 : s2.end();

      // Add the is_managed flag after configuring package policies to avoid errors
      if (shouldAddIsManagedFlag) {
        await _agent_policy.agentPolicyService.update(namespacedSoClient, esClient, policy.id, {
          is_managed: true
        }, {
          force: true,
          bumpRevision: false
        });
      }
      if (packagePoliciesToAdd.length > 0 || packagePoliciesUpdates.length > 0 || shouldAddIsManagedFlag) {
        await _agent_policy.agentPolicyService.bumpRevision(namespacedSoClient, esClient, policy.id);
      }
    }
  }
  return {
    policies: fulfilledPolicies.map(p => p.policy ? {
      id: p.policy.id,
      updated_at: p.policy.updated_at
    } : {
      id: p.deleted,
      updated_at: _i18n.i18n.translate('xpack.fleet.preconfiguration.policyDeleted', {
        defaultMessage: 'Preconfigured policy {id} was deleted; skipping creation',
        values: {
          id: p.deleted
        }
      })
    }),
    // @ts-expect-error upgrade typescript v4.9.5
    packages: fulfilledPackages.map(pkg => 'version' in pkg ? (0, _registry.pkgToPkgKey)(pkg) : pkg.name),
    nonFatalErrors: [...rejectedPackages, ...rejectedPolicies]
  };
}
function comparePreconfiguredPolicyToCurrent(policyFromConfig, currentPolicy) {
  // Namespace is omitted from being compared because even for managed policies, we still
  // want users to be able to pick their own namespace: https://github.com/elastic/kibana/issues/110533
  const configTopLevelFields = (0, _lodash.omit)(policyFromConfig, 'package_policies', 'id', 'namespace', 'space_id');
  const currentTopLevelFields = (0, _lodash.pick)(currentPolicy, ...Object.keys(configTopLevelFields));
  return {
    hasChanged: !(0, _lodash.isEqual)(configTopLevelFields, currentTopLevelFields),
    fields: configTopLevelFields
  };
}
async function addPreconfiguredPolicyPackages(esClient, agentPolicy, installedPackagePolicies, defaultOutput, bumpAgentPolicyRevison = false) {
  // Cache package info objects so we don't waste lookup time on the latest package
  // every time we call `getPackageInfo`
  const packageInfoMap = new Map();

  // Add packages synchronously to avoid overwriting
  for (const {
    installedPackage,
    packagePolicy,
    namespacedSoClient
  } of installedPackagePolicies) {
    let packageInfo;
    if (packageInfoMap.has(installedPackage.name)) {
      packageInfo = packageInfoMap.get(installedPackage.name);
    } else {
      packageInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: namespacedSoClient,
        pkgName: installedPackage.name,
        pkgVersion: installedPackage.version
      });
    }
    if (Array.isArray(packagePolicy.inputs)) {
      const {
        id,
        name,
        description,
        inputs
      } = packagePolicy;
      await (0, _agent_policy.addPackageToAgentPolicy)(namespacedSoClient, esClient, agentPolicy, packageInfo, name, id, description, policy => (0, _package_policy.preconfigurePackageInputs)(policy, packageInfo, inputs), bumpAgentPolicyRevison);
    } else {
      var _simplifiedPackagePol;
      const simplifiedPackagePolicy = packagePolicy;
      const id = (_simplifiedPackagePol = simplifiedPackagePolicy.id) === null || _simplifiedPackagePol === void 0 ? void 0 : _simplifiedPackagePol.toString();
      // Simplified package policy
      const newPackagePolicy = (0, _simplified_package_policy_helper.simplifiedPackagePolicytoNewPackagePolicy)({
        ...simplifiedPackagePolicy,
        id,
        policy_id: agentPolicy.id,
        policy_ids: [agentPolicy.id],
        namespace: packagePolicy.namespace || agentPolicy.namespace
      }, packageInfo, {});
      await _package_policy.packagePolicyService.create(namespacedSoClient, esClient, newPackagePolicy, {
        id,
        bumpRevision: bumpAgentPolicyRevison,
        skipEnsureInstalled: true,
        skipUniqueNameVerification: true,
        overwrite: true,
        force: true,
        // To add package to managed policy we need the force flag
        packageInfo
      });
    }
  }
}