"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stepCreateRestartInstallation = stepCreateRestartInstallation;
var _gt = _interopRequireDefault(require("semver/functions/gt"));
var _errors = require("../../../../../errors");
var _constants = require("../../../../../constants");
var _install = require("../../install");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function stepCreateRestartInstallation(context) {
  const {
    savedObjectsClient,
    logger,
    installSource,
    packageInstallContext,
    spaceId,
    force,
    verificationResult,
    installedPkg
  } = context;
  const {
    packageInfo
  } = packageInstallContext;
  const {
    name: pkgName,
    version: pkgVersion
  } = packageInfo;

  // if some installation already exists
  if (installedPkg) {
    const previousVersion = (0, _gt.default)(pkgVersion, installedPkg.attributes.install_version) ? installedPkg.attributes.install_version : undefined;
    const isStatusInstalling = installedPkg.attributes.install_status === 'installing';
    const hasExceededTimeout = Date.now() - Date.parse(installedPkg.attributes.install_started_at) < _constants.MAX_TIME_COMPLETE_INSTALL;
    logger.debug(`Package install - Install status ${installedPkg.attributes.install_status}`);

    // if the installation is currently running, don't try to install
    // instead, only return already installed assets
    if (isStatusInstalling && hasExceededTimeout) {
      // If this is a forced installation, ignore the timeout and restart the installation anyway
      logger.debug(`Package install - Installation is running and has exceeded timeout`);
      if (force) {
        logger.debug(`Package install - Forced installation, restarting`);
        await (0, _utils.withPackageSpan)('Restarting installation with force flag', () => (0, _install.restartInstallation)({
          savedObjectsClient,
          pkgName,
          pkgVersion,
          installSource,
          verificationResult,
          previousVersion
        }));
      } else {
        throw new _errors.ConcurrentInstallOperationError(`Concurrent installation or upgrade of ${pkgName || 'unknown'}-${pkgVersion || 'unknown'} detected, aborting.`);
      }
    } else {
      // if no installation is running, or the installation has been running longer than MAX_TIME_COMPLETE_INSTALL
      // (it might be stuck) update the saved object and proceed
      logger.debug(`Package install - no installation running or the installation has been running longer than ${_constants.MAX_TIME_COMPLETE_INSTALL}, restarting`);
      await (0, _utils.withPackageSpan)('Restarting installation', () => (0, _install.restartInstallation)({
        savedObjectsClient,
        pkgName,
        pkgVersion,
        installSource,
        verificationResult,
        previousVersion
      }));
    }
  } else {
    logger.debug(`Package install - Create installation`);
    await (0, _utils.withPackageSpan)('Creating installation', () => (0, _install.createInstallation)({
      savedObjectsClient,
      packageInfo,
      installSource,
      spaceId,
      verificationResult
    }));
  }
}