"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installKibanaAssetsWithStreaming = installKibanaAssetsWithStreaming;
var _archive = require("../../archive");
var _install = require("../../packages/install");
var _install2 = require("./install");
var _saved_objects = require("./saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ASSETS_TO_INSTALL_IN_PARALLEL = 100;
async function installKibanaAssetsWithStreaming({
  spaceId,
  packageInstallContext,
  savedObjectsClient,
  pkgName
}) {
  const {
    archiveIterator
  } = packageInstallContext;
  const {
    savedObjectClientWithSpace,
    savedObjectsImporter
  } = (0, _saved_objects.getSpaceAwareSaveobjectsClients)(spaceId);
  await (0, _install2.installManagedIndexPattern)({
    savedObjectsImporter,
    savedObjectsClient
  });
  const assetRefs = [];
  let batch = [];
  await archiveIterator.traverseEntries(async ({
    path,
    buffer
  }) => {
    if (!buffer || !(0, _install2.isKibanaAssetType)(path)) {
      return;
    }
    const savedObject = JSON.parse(buffer.toString('utf8'));
    const assetType = (0, _archive.getPathParts)(path).type;
    const soType = _install2.KibanaSavedObjectTypeMapping[assetType];
    if (savedObject.type !== soType) {
      return;
    }
    batch.push(savedObject);
    assetRefs.push((0, _install2.toAssetReference)(savedObject));
    if (batch.length >= MAX_ASSETS_TO_INSTALL_IN_PARALLEL) {
      await bulkCreateSavedObjects({
        savedObjectsClient: savedObjectClientWithSpace,
        kibanaAssets: batch,
        refresh: false
      });
      batch = [];
    }
  });

  // install any remaining assets
  if (batch.length) {
    await bulkCreateSavedObjects({
      savedObjectsClient: savedObjectClientWithSpace,
      kibanaAssets: batch,
      // Use wait_for with the last batch to ensure all assets are readable once the install is complete
      refresh: 'wait_for'
    });
  }

  // Update the installation saved object with installed kibana assets
  await (0, _install.saveKibanaAssetsRefs)(savedObjectsClient, pkgName, assetRefs);
  return assetRefs;
}
async function bulkCreateSavedObjects({
  savedObjectsClient,
  kibanaAssets,
  refresh
}) {
  if (!kibanaAssets.length) {
    return [];
  }
  const toBeSavedObjects = kibanaAssets.map(asset => (0, _install2.createSavedObjectKibanaAsset)(asset));
  const {
    saved_objects: createdSavedObjects
  } = await savedObjectsClient.bulkCreate(toBeSavedObjects, {
    // We only want to install new saved objects without overwriting existing ones
    overwrite: false,
    managed: true,
    refresh
  });
  return createdSavedObjects;
}