"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.catchAndSetErrorStackTrace = void 0;
exports.isESClientError = isESClientError;
exports.isElasticsearchVersionConflictError = isElasticsearchVersionConflictError;
var _elasticsearch = require("@elastic/elasticsearch");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isESClientError(error) {
  return error instanceof _elasticsearch.errors.ResponseError;
}
function isElasticsearchVersionConflictError(error) {
  return isESClientError(error) && error.meta.statusCode === 409;
}
/**
 * Error handling utility for use with promises that will set the stack trace on the error provided.
 * Especially useful when working with ES/SO client, as errors thrown by those client normally do
 * not include a very helpful stack trace.
 *
 * @param error
 * @param message
 *
 * @example
 *
 *    esClient.search(...).catch(catchAndSetErrorStackTrace);
 *
 *    // With custom message on error thrown
 *    esClient.search(...).catch(catchAndSetErrorStackTrace.withMessage('update to item xyz failed'));
 *
 */
const catchAndSetErrorStackTrace = (error, message = '') => {
  const priorStackTrace = error.stack;
  Error.captureStackTrace(error, catchAndSetErrorStackTrace);
  error.stack += `\n----[ ORIGINAL STACK TRACE ]----\n${priorStackTrace}`;
  if (message) {
    error.stack = message + '\n' + error.stack;
  }
  return Promise.reject(error);
};
exports.catchAndSetErrorStackTrace = catchAndSetErrorStackTrace;
catchAndSetErrorStackTrace.withMessage = message => {
  return err => catchAndSetErrorStackTrace(err, message);
};