"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateAndUpdateAttackDiscoveries = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _deduplication = require("../../../lib/attack_discovery/persistence/deduplication");
var _helpers = require("./helpers");
var _handle_graph_error = require("../post/helpers/handle_graph_error");
var _generate_discoveries = require("./generate_discoveries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const generateAndUpdateAttackDiscoveries = async ({
  actionsClient,
  attackDiscoveryAlertsEnabled,
  authenticatedUser,
  config,
  dataClient,
  esClient,
  executionUuid,
  logger,
  savedObjectsClient,
  telemetry
}) => {
  const startTime = (0, _moment.default)(); // start timing the generation

  // get parameters from the request body
  const {
    apiConfig,
    connectorName,
    end,
    filter,
    replacements,
    size,
    start
  } = config;
  let latestReplacements = {
    ...replacements
  };
  try {
    const {
      anonymizedAlerts,
      attackDiscoveries,
      replacements: generatedReplacements
    } = await (0, _generate_discoveries.generateAttackDiscoveries)({
      actionsClient,
      config,
      esClient,
      logger,
      savedObjectsClient
    });
    latestReplacements = generatedReplacements;
    await (0, _helpers.updateAttackDiscoveries)({
      anonymizedAlerts,
      apiConfig,
      attackDiscoveries,
      executionUuid,
      authenticatedUser,
      dataClient,
      hasFilter: !!(filter && Object.keys(filter).length),
      end,
      latestReplacements,
      logger,
      size,
      start,
      startTime,
      telemetry
    });
    let storedAttackDiscoveries = attackDiscoveries;
    if (attackDiscoveryAlertsEnabled) {
      var _authenticatedUser$us;
      const alertsContextCount = anonymizedAlerts.length;

      /**
       * Deduplicate attackDiscoveries before creating alerts
       *
       * We search for duplicates within the ad hoc index only,
       * because there will be no duplicates in the scheduled index due to the
       * fact that we use schedule ID (for the schedules) and
       * user ID (for the ad hoc generations) as part of the alert ID hash
       * generated for the deduplication purposes
       */
      const indexPattern = dataClient.getAdHocAlertsIndexPattern();
      const dedupedDiscoveries = await (0, _deduplication.deduplicateAttackDiscoveries)({
        esClient,
        attackDiscoveries: attackDiscoveries !== null && attackDiscoveries !== void 0 ? attackDiscoveries : [],
        connectorId: apiConfig.connectorId,
        indexPattern,
        logger,
        ownerInfo: {
          id: (_authenticatedUser$us = authenticatedUser.username) !== null && _authenticatedUser$us !== void 0 ? _authenticatedUser$us : authenticatedUser.profile_uid,
          isSchedule: false
        },
        replacements: latestReplacements,
        spaceId: dataClient.spaceId
      });
      storedAttackDiscoveries = dedupedDiscoveries;
      const createAttackDiscoveryAlertsParams = {
        alertsContextCount,
        anonymizedAlerts,
        apiConfig,
        attackDiscoveries: dedupedDiscoveries,
        connectorName: connectorName !== null && connectorName !== void 0 ? connectorName : apiConfig.connectorId,
        generationUuid: executionUuid,
        replacements: latestReplacements
      };
      await dataClient.createAttackDiscoveryAlerts({
        authenticatedUser,
        createAttackDiscoveryAlertsParams
      });
    }
    return {
      anonymizedAlerts,
      attackDiscoveries: storedAttackDiscoveries,
      replacements: latestReplacements
    };
  } catch (err) {
    await (0, _handle_graph_error.handleGraphError)({
      apiConfig,
      executionUuid,
      authenticatedUser,
      dataClient,
      err,
      latestReplacements,
      logger,
      telemetry
    });
    return {
      error: err
    };
  }
};
exports.generateAndUpdateAttackDiscoveries = generateAndUpdateAttackDiscoveries;