"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToUpdateScheme = exports.transformToCreateScheme = exports.transformESToAlertSummary = exports.transformESSearchToAlertSummary = exports.getUpdateScript = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const transformESToAlertSummary = response => {
  return response.map(alertSummarySchema => {
    var _alertSummarySchema$u, _alertSummarySchema$u2;
    const alertSummary = {
      timestamp: alertSummarySchema['@timestamp'],
      createdAt: alertSummarySchema.created_at,
      users: (_alertSummarySchema$u = (_alertSummarySchema$u2 = alertSummarySchema.users) === null || _alertSummarySchema$u2 === void 0 ? void 0 : _alertSummarySchema$u2.map(user => ({
        id: user.id,
        name: user.name
      }))) !== null && _alertSummarySchema$u !== void 0 ? _alertSummarySchema$u : [],
      summary: alertSummarySchema.summary,
      alertId: alertSummarySchema.alert_id,
      updatedAt: alertSummarySchema.updated_at,
      recommendedActions: alertSummarySchema.recommended_actions,
      namespace: alertSummarySchema.namespace,
      id: alertSummarySchema.id,
      createdBy: alertSummarySchema.created_by,
      updatedBy: alertSummarySchema.updated_by,
      replacements: alertSummarySchema.replacements.reduce((acc, r) => {
        acc[r.uuid] = r.value;
        return acc;
      }, {})
    };
    return alertSummary;
  });
};
exports.transformESToAlertSummary = transformESToAlertSummary;
const transformESSearchToAlertSummary = response => {
  return response.hits.hits.filter(hit => hit._source !== undefined).map(hit => {
    var _alertSummarySchema$u3, _alertSummarySchema$u4;
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    const alertSummarySchema = hit._source;
    const alertSummary = {
      timestamp: alertSummarySchema['@timestamp'],
      createdAt: alertSummarySchema.created_at,
      users: (_alertSummarySchema$u3 = (_alertSummarySchema$u4 = alertSummarySchema.users) === null || _alertSummarySchema$u4 === void 0 ? void 0 : _alertSummarySchema$u4.map(user => ({
        id: user.id,
        name: user.name
      }))) !== null && _alertSummarySchema$u3 !== void 0 ? _alertSummarySchema$u3 : [],
      summary: alertSummarySchema.summary,
      recommendedActions: alertSummarySchema.recommended_actions,
      updatedAt: alertSummarySchema.updated_at,
      namespace: alertSummarySchema.namespace,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      id: hit._id,
      alertId: alertSummarySchema.alert_id,
      createdBy: alertSummarySchema.created_by,
      updatedBy: alertSummarySchema.updated_by,
      replacements: alertSummarySchema.replacements.reduce((acc, r) => {
        acc[r.uuid] = r.value;
        return acc;
      }, {})
    };
    return alertSummary;
  });
};
exports.transformESSearchToAlertSummary = transformESSearchToAlertSummary;
const transformToUpdateScheme = (user, updatedAt, {
  summary,
  recommendedActions,
  replacements,
  id
}) => {
  return {
    id,
    updated_at: updatedAt,
    updated_by: user.username,
    ...(summary ? {
      summary
    } : {}),
    ...(recommendedActions ? {
      recommended_actions: recommendedActions
    } : {}),
    ...(replacements ? {
      replacements: Object.keys(replacements).map(key => ({
        uuid: key,
        value: replacements[key]
      }))
    } : {})
  };
};
exports.transformToUpdateScheme = transformToUpdateScheme;
const transformToCreateScheme = (user, updatedAt, {
  summary,
  alertId,
  recommendedActions,
  replacements
}) => {
  return {
    '@timestamp': updatedAt,
    updated_at: updatedAt,
    updated_by: user.username,
    created_at: updatedAt,
    created_by: user.username,
    summary: summary !== null && summary !== void 0 ? summary : '',
    ...(recommendedActions ? {
      recommended_actions: recommendedActions
    } : {}),
    alert_id: alertId,
    users: [{
      id: user.profile_uid,
      name: user.username
    }],
    replacements: Object.keys(replacements).map(key => ({
      uuid: key,
      value: replacements[key]
    }))
  };
};
exports.transformToCreateScheme = transformToCreateScheme;
const getUpdateScript = ({
  alertSummary,
  isPatch
}) => {
  return {
    script: {
      source: `
    if (params.assignEmpty == true || params.containsKey('summary')) {
      ctx._source.summary = params.summary;
    }
    if (params.assignEmpty == true || params.containsKey('recommended_actions')) {
      ctx._source.recommended_actions = params.recommended_actions;
    }
    ctx._source.updated_at = params.updated_at;
  `,
      lang: 'painless',
      params: {
        ...alertSummary,
        // when assigning undefined in painless, it will remove property and wil set it to null
        // for patch we don't want to remove unspecified value in payload
        assignEmpty: !(isPatch !== null && isPatch !== void 0 ? isPatch : true)
      }
    }
  };
};
exports.getUpdateScript = getUpdateScript;