"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDeleteScriptedFieldRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _error = require("../../../error");
var _handle_errors = require("../util/handle_errors");
var _constants = require("../../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerDeleteScriptedFieldRoute = (router, getStartServices) => {
  router.versioned.delete({
    path: '/api/index_patterns/index_pattern/{id}/scripted_field/{name}',
    access: 'public',
    security: {
      authz: {
        requiredPrivileges: ['indexPatterns:manage']
      }
    }
  }).addVersion({
    version: _constants.INITIAL_REST_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          }),
          name: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          })
        }, {
          unknowns: 'allow'
        })
      },
      response: {
        200: {
          body: () => _configSchema.schema.never()
        }
      }
    }
  }, router.handleLegacyErrors((0, _handle_errors.handleErrors)(async (ctx, req, res) => {
    const core = await ctx.core;
    const savedObjectsClient = core.savedObjects.client;
    const elasticsearchClient = core.elasticsearch.client.asCurrentUser;
    const [,, {
      dataViewsServiceFactory
    }] = await getStartServices();
    const indexPatternsService = await dataViewsServiceFactory(savedObjectsClient, elasticsearchClient, req);
    const id = req.params.id;
    const name = req.params.name;
    const indexPattern = await indexPatternsService.getDataViewLazy(id);
    const field = await indexPattern.getFieldByName(name);
    if (!field) {
      throw new _error.ErrorIndexPatternFieldNotFound(id, name);
    }
    if (!field.scripted) {
      throw new Error('Only scripted fields can be deleted.');
    }
    indexPattern.removeScriptedField(name);
    await indexPatternsService.updateSavedObject(indexPattern);
    return res.ok({
      headers: {
        'content-type': 'application/json'
      }
    });
  })));
};
exports.registerDeleteScriptedFieldRoute = registerDeleteScriptedFieldRoute;