"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.JavaScriptExporter = void 0;
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
const handlebars_1 = __importDefault(require("handlebars"));
const prettier_1 = __importDefault(require("prettier"));
const parser_typescript_1 = __importDefault(require("prettier/parser-typescript"));
require("./templates");
const UNSUPPORTED_APIS = new RegExp("^query_rules.*$" +
    "|^connector.update_features$" +
    "|^connector.sync_job_.*$" +
    "|^ingest.get_geoip_database$" +
    "|^ingest.put_geoip_database$" +
    "|^ingest.delete_geoip_database$" +
    "|^_internal.*$" +
    "|^security.create_cross_cluster_api_key$" +
    "|^security.update_cross_cluster_api_key$" +
    "|^security.update_settings$" +
    "|^snapshot.repository_analyze$" +
    "|^watcher.get_settings$" +
    "|^watcher.update_settings");
class JavaScriptExporter {
    async check(requests) {
        // only return true if all requests are for Elasticsearch
        return requests
            .map((req) => req.service == "es")
            .reduce((prev, curr) => prev && curr, true);
    }
    async convert(requests, options) {
        if (!(await this.check(requests))) {
            throw new Error("Cannot perform conversion");
        }
        const output = this.template({ requests, ...options });
        return prettier_1.default.format(output, {
            parser: "typescript",
            plugins: [parser_typescript_1.default],
        });
    }
    get template() {
        if (!this._template) {
            handlebars_1.default.registerHelper("json", function (context) {
                const val = JSON.stringify(context ?? null, null, 2);
                // turn number strings into numbers
                if (val.match(/^"\d+"$/)) {
                    return parseInt(val.replaceAll('"', ""), 10);
                }
                return val;
            });
            // custom conditional for requests without any arguments
            handlebars_1.default.registerHelper("hasArgs", function (options) {
                if (Object.keys(this.params ?? {}).length +
                    Object.keys(this.query ?? {}).length +
                    Object.keys(this.body ?? {}).length >
                    0) {
                    return options.fn(this);
                }
                else {
                    return options.inverse(this);
                }
            });
            // custom conditional to separate supported vs unsupported APIs
            handlebars_1.default.registerHelper("supportedApi", function (options) {
                if (!UNSUPPORTED_APIS.test(this.api) && this.request) {
                    return options.fn(this);
                }
                else {
                    return options.inverse(this);
                }
            });
            // attribute name renderer that considers aliases and code-specific names
            // arguments:
            //   name: the name of the attribute
            //   props: the list of schema properties this attribute belongs to
            handlebars_1.default.registerHelper("alias", (name, props) => {
                if (props) {
                    for (const prop of props) {
                        if (prop.name == name && prop.codegenName != undefined) {
                            return prop.codegenName;
                        }
                    }
                }
                return name;
            });
            // custom conditional to check for request body kind
            // the argument can be "properties" or "value"
            handlebars_1.default.registerHelper("ifRequestBodyKind", function (kind, options) {
                const bodyKind = this.request?.body?.kind ?? "value";
                if (bodyKind == kind) {
                    return options.fn(this);
                }
                else {
                    return options.inverse(this);
                }
            });
            handlebars_1.default.registerHelper("camelCase", (text) => toCamelCase(text));
            if (process.env.NODE_ENV !== "test") {
                this._template = handlebars_1.default.templates["javascript.tpl"];
            }
            else {
                // when running tests we read the templates directly, in case the
                // compiled file is not up to date
                const t = (0, fs_1.readFileSync)(path_1.default.join(__dirname, "./javascript.tpl"), "utf-8");
                this._template = handlebars_1.default.compile(t);
            }
        }
        return this._template;
    }
}
exports.JavaScriptExporter = JavaScriptExporter;
function toCamelCase(str) {
    return str.replace(/_([a-z])/g, (k) => k[1].toUpperCase());
}
