"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createNewTermsAlertType = void 0;
var _lodash = require("lodash");
var _securitysolutionRules = require("@kbn/securitysolution-rules");
var _coreApplicationCommon = require("@kbn/core-application-common");
var _constants = require("../../../../../common/constants");
var _rule_schema = require("../../rule_schema");
var _single_search_after = require("../utils/single_search_after");
var _build_events_query = require("../utils/build_events_query");
var _get_filter = require("../utils/get_filter");
var _wrap_new_terms_alerts = require("./wrap_new_terms_alerts");
var _bulk_create_suppressed_alerts_in_memory = require("./bulk_create_suppressed_alerts_in_memory");
var _build_new_terms_aggregation = require("./build_new_terms_aggregation");
var _utils = require("../utils");
var _utils2 = require("./utils");
var _utils3 = require("../utils/utils");
var _get_is_alert_suppression_active = require("../utils/get_is_alert_suppression_active");
var _multi_terms_composite = require("./multi_terms_composite");
var i18n = _interopRequireWildcard(require("../translations"));
var _factories = require("../factories");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createNewTermsAlertType = () => {
  return {
    id: _securitysolutionRules.NEW_TERMS_RULE_TYPE_ID,
    name: 'New Terms Rule',
    validate: {
      params: {
        validate: object => {
          const validated = _rule_schema.NewTermsRuleParams.parse(object);
          (0, _utils2.validateHistoryWindowStart)({
            historyWindowStart: validated.historyWindowStart,
            from: validated.from
          });
          return validated;
        },
        /**
         * validate rule params when rule is bulk edited (update and created in future as well)
         * returned params can be modified (useful in case of version increment)
         * @param mutatedRuleParams
         * @returns mutatedRuleParams
         */
        validateMutatedParams: mutatedRuleParams => {
          (0, _utils.validateIndexPatterns)(mutatedRuleParams.index);
          return mutatedRuleParams;
        }
      }
    },
    schemas: {
      params: {
        type: 'zod',
        schema: _rule_schema.NewTermsRuleParams
      }
    },
    actionGroups: [{
      id: 'default',
      name: 'Default'
    }],
    defaultActionGroupId: 'default',
    actionVariables: {
      context: [{
        name: 'server',
        description: 'the server'
      }]
    },
    minimumLicenseRequired: 'basic',
    isExportable: false,
    category: _coreApplicationCommon.DEFAULT_APP_CATEGORIES.security.id,
    producer: _constants.SERVER_APP_ID,
    solution: 'security',
    async executor(execOptions) {
      const {
        sharedParams,
        services,
        params,
        state,
        logger
      } = execOptions;
      const {
        ruleExecutionLogger,
        completeRule,
        tuple,
        inputIndex,
        runtimeMappings,
        primaryTimestamp,
        secondaryTimestamp,
        aggregatableTimestampField,
        exceptionFilter,
        unprocessedExceptions,
        licensing,
        scheduleNotificationResponseActionsService
      } = sharedParams;
      const isLoggedRequestsEnabled = Boolean(state === null || state === void 0 ? void 0 : state.isLoggedRequestsEnabled);
      const loggedRequests = [];

      // Validate the history window size compared to `from` at runtime as well as in the `validate`
      // function because rule preview does not use the `validate` function defined on the rule type
      (0, _utils2.validateHistoryWindowStart)({
        historyWindowStart: params.historyWindowStart,
        from: params.from
      });
      const filterArgs = {
        filters: params.filters,
        index: inputIndex,
        language: params.language,
        savedId: undefined,
        services,
        type: params.type,
        query: params.query,
        exceptionFilter,
        loadFields: true
      };
      const esFilter = await (0, _get_filter.getFilter)(filterArgs);
      const parsedHistoryWindowSize = (0, _utils2.parseDateString)({
        date: params.historyWindowStart,
        forceNow: tuple.to.toDate(),
        name: 'historyWindowStart'
      });
      const isAlertSuppressionActive = await (0, _get_is_alert_suppression_active.getIsAlertSuppressionActive)({
        alertSuppression: params.alertSuppression,
        licensing
      });
      let afterKey;
      const result = (0, _utils3.createSearchAfterReturnType)();
      const exceptionsWarning = (0, _utils3.getUnprocessedExceptionsWarnings)(unprocessedExceptions);
      if (exceptionsWarning) {
        result.warningMessages.push(exceptionsWarning);
      }
      let pageNumber = 0;

      // There are 2 conditions that mean we're finished: either there were still too many alerts to create
      // after deduplication and the array of alerts was truncated before being submitted to ES, or there were
      // exactly enough new alerts to hit maxSignals without truncating the array of alerts. We check both because
      // it's possible for the array to be truncated but alert documents could fail to be created for other reasons,
      // in which case createdSignalsCount would still be less than maxSignals. Since valid alerts were truncated from
      // the array in that case, we stop and report the errors.
      while (result.createdSignalsCount <= params.maxSignals) {
        pageNumber++;
        // PHASE 1: Fetch a page of terms using a composite aggregation. This will collect a page from
        // all of the terms seen over the last rule interval. In the next phase we'll determine which
        // ones are new.
        const searchRequest = (0, _build_events_query.buildEventsSearchQuery)({
          aggregations: (0, _build_new_terms_aggregation.buildRecentTermsAgg)({
            fields: params.newTermsFields,
            after: afterKey
          }),
          searchAfterSortIds: undefined,
          index: inputIndex,
          // The time range for the initial composite aggregation is the rule interval, `from` and `to`
          from: tuple.from.toISOString(),
          to: tuple.to.toISOString(),
          filter: esFilter,
          size: 0,
          primaryTimestamp,
          secondaryTimestamp,
          runtimeMappings
        });
        const {
          searchResult,
          searchDuration,
          searchErrors,
          loggedRequests: firstPhaseLoggedRequests = []
        } = await (0, _single_search_after.singleSearchAfter)({
          searchRequest,
          services,
          ruleExecutionLogger,
          loggedRequestsConfig: isLoggedRequestsEnabled ? {
            type: 'findAllTerms',
            description: i18n.FIND_ALL_NEW_TERMS_FIELDS_DESCRIPTION((0, _utils3.stringifyAfterKey)(afterKey)),
            skipRequestQuery: pageNumber > 2
          } : undefined
        });
        loggedRequests.push(...firstPhaseLoggedRequests);
        if (!searchResult.aggregations) {
          throw new Error('Aggregations were missing on recent terms search result');
        }
        logger.debug(`Time spent on composite agg: ${searchDuration}`);
        result.searchAfterTimes.push(searchDuration);
        result.errors.push(...searchErrors);

        // If the aggregation returns no after_key it signals that we've paged through all results
        // and the current page is empty so we can immediately break.
        if (searchResult.aggregations.new_terms.after_key == null) {
          break;
        }
        const bucketsForField = searchResult.aggregations.new_terms.buckets;
        const createAlertsHook = async aggResult => {
          var _aggResult$aggregatio, _aggResult$aggregatio2;
          const eventsAndTerms = ((_aggResult$aggregatio = aggResult === null || aggResult === void 0 ? void 0 : (_aggResult$aggregatio2 = aggResult.aggregations) === null || _aggResult$aggregatio2 === void 0 ? void 0 : _aggResult$aggregatio2.new_terms.buckets) !== null && _aggResult$aggregatio !== void 0 ? _aggResult$aggregatio : []).map(bucket => {
            const newTerms = (0, _lodash.isObject)(bucket.key) ? Object.values(bucket.key) : [bucket.key];
            return {
              event: bucket.docs.hits.hits[0],
              newTerms
            };
          });
          let bulkCreateResult = {
            errors: [],
            success: true,
            enrichmentDuration: '0',
            bulkCreateDuration: '0',
            createdItemsCount: 0,
            createdItems: [],
            alertsWereTruncated: false
          };

          // wrap and create alerts by chunks
          // large number of matches, processed in possibly 10,000 size of events and terms
          // can significantly affect Kibana performance
          const eventAndTermsChunks = (0, _lodash.chunk)(eventsAndTerms, 5 * params.maxSignals);
          for (let i = 0; i < eventAndTermsChunks.length; i++) {
            const eventAndTermsChunk = eventAndTermsChunks[i];
            if (isAlertSuppressionActive && (0, _get_is_alert_suppression_active.alertSuppressionTypeGuard)(params.alertSuppression)) {
              bulkCreateResult = await (0, _bulk_create_suppressed_alerts_in_memory.bulkCreateSuppressedNewTermsAlertsInMemory)({
                sharedParams,
                eventsAndTerms: eventAndTermsChunk,
                toReturn: result,
                services,
                alertSuppression: params.alertSuppression
              });
            } else {
              const wrappedAlerts = (0, _wrap_new_terms_alerts.wrapNewTermsAlerts)({
                sharedParams,
                eventsAndTerms: eventAndTermsChunk
              });
              bulkCreateResult = await (0, _factories.bulkCreate)({
                wrappedAlerts,
                services,
                sharedParams,
                maxAlerts: params.maxSignals - result.createdSignalsCount
              });
              (0, _utils3.addToSearchAfterReturn)({
                current: result,
                next: bulkCreateResult
              });
            }
            if (bulkCreateResult.alertsWereTruncated) {
              break;
            }
          }
          return bulkCreateResult;
        };

        // separate route for multiple new terms
        // it uses paging through composite aggregation
        if (params.newTermsFields.length > 1) {
          var _bulkCreateResult$log;
          const bulkCreateResult = await (0, _multi_terms_composite.multiTermsComposite)({
            sharedParams,
            filterArgs,
            buckets: bucketsForField,
            params,
            aggregatableTimestampField,
            parsedHistoryWindowSize,
            services,
            result,
            logger,
            afterKey,
            createAlertsHook,
            isAlertSuppressionActive,
            isLoggedRequestsEnabled
          });
          loggedRequests.push(...((_bulkCreateResult$log = bulkCreateResult === null || bulkCreateResult === void 0 ? void 0 : bulkCreateResult.loggedRequests) !== null && _bulkCreateResult$log !== void 0 ? _bulkCreateResult$log : []));
          if (bulkCreateResult && 'alertsWereTruncated' in bulkCreateResult) {
            break;
          }
        } else {
          // PHASE 2: Take the page of results from Phase 1 and determine if each term exists in the history window.
          // The aggregation filters out buckets for terms that exist prior to `tuple.from`, so the buckets in the
          // response correspond to each new term.
          const includeValues = (0, _utils2.transformBucketsToValues)(params.newTermsFields, bucketsForField);
          const pageSearchRequest = (0, _build_events_query.buildEventsSearchQuery)({
            aggregations: (0, _build_new_terms_aggregation.buildNewTermsAgg)({
              newValueWindowStart: tuple.from,
              timestampField: aggregatableTimestampField,
              field: params.newTermsFields[0],
              include: includeValues
            }),
            runtimeMappings,
            searchAfterSortIds: undefined,
            index: inputIndex,
            // For Phase 2, we expand the time range to aggregate over the history window
            // in addition to the rule interval
            from: parsedHistoryWindowSize.toISOString(),
            to: tuple.to.toISOString(),
            filter: esFilter,
            size: 0,
            primaryTimestamp,
            secondaryTimestamp
          });
          const {
            searchResult: pageSearchResult,
            searchDuration: pageSearchDuration,
            searchErrors: pageSearchErrors,
            loggedRequests: pageSearchLoggedRequests = []
          } = await (0, _single_search_after.singleSearchAfter)({
            searchRequest: pageSearchRequest,
            services,
            ruleExecutionLogger,
            loggedRequestsConfig: isLoggedRequestsEnabled ? {
              type: 'findNewTerms',
              description: i18n.FIND_NEW_TERMS_VALUES_DESCRIPTION((0, _utils3.stringifyAfterKey)(afterKey)),
              skipRequestQuery: pageNumber > 2
            } : undefined
          });
          result.searchAfterTimes.push(pageSearchDuration);
          result.errors.push(...pageSearchErrors);
          loggedRequests.push(...pageSearchLoggedRequests);
          logger.debug(`Time spent on phase 2 terms agg: ${pageSearchDuration}`);
          if (!pageSearchResult.aggregations) {
            throw new Error('Aggregations were missing on new terms search result');
          }

          // PHASE 3: For each term that is not in the history window, fetch the oldest document in
          // the rule interval for that term. This is the first document to contain the new term, and will
          // become the basis of the resulting alert.
          // One document could become multiple alerts if the document contains an array with multiple new terms.
          if (pageSearchResult.aggregations.new_terms.buckets.length > 0) {
            const actualNewTerms = pageSearchResult.aggregations.new_terms.buckets.map(bucket => bucket.key);
            const docFetchSearchRequest = (0, _build_events_query.buildEventsSearchQuery)({
              aggregations: (0, _build_new_terms_aggregation.buildDocFetchAgg)({
                timestampField: aggregatableTimestampField,
                field: params.newTermsFields[0],
                include: actualNewTerms
              }),
              runtimeMappings,
              searchAfterSortIds: undefined,
              index: inputIndex,
              // For phase 3, we go back to aggregating only over the rule interval - excluding the history window
              from: tuple.from.toISOString(),
              to: tuple.to.toISOString(),
              filter: esFilter,
              size: 0,
              primaryTimestamp,
              secondaryTimestamp
            });
            const {
              searchResult: docFetchSearchResult,
              searchDuration: docFetchSearchDuration,
              searchErrors: docFetchSearchErrors,
              loggedRequests: docFetchLoggedRequests = []
            } = await (0, _single_search_after.singleSearchAfter)({
              searchRequest: docFetchSearchRequest,
              services,
              ruleExecutionLogger,
              loggedRequestsConfig: isLoggedRequestsEnabled ? {
                type: 'findDocuments',
                description: i18n.FIND_NEW_TERMS_EVENTS_DESCRIPTION((0, _utils3.stringifyAfterKey)(afterKey)),
                skipRequestQuery: pageNumber > 2
              } : undefined
            });
            result.searchAfterTimes.push(docFetchSearchDuration);
            result.errors.push(...docFetchSearchErrors);
            loggedRequests.push(...docFetchLoggedRequests);
            if (!docFetchSearchResult.aggregations) {
              throw new Error('Aggregations were missing on document fetch search result');
            }
            const bulkCreateResult = await createAlertsHook(docFetchSearchResult);
            if (bulkCreateResult.alertsWereTruncated) {
              result.warningMessages.push(isAlertSuppressionActive ? (0, _utils3.getSuppressionMaxSignalsWarning)() : (0, _utils3.getMaxSignalsWarning)());
              break;
            }
          }
        }
        afterKey = searchResult.aggregations.new_terms.after_key;
      }
      scheduleNotificationResponseActionsService({
        signals: result.createdSignals,
        signalsCount: result.createdSignalsCount,
        responseActions: completeRule.ruleParams.responseActions
      });
      return {
        ...result,
        state,
        ...(isLoggedRequestsEnabled ? {
          loggedRequests
        } : {})
      };
    }
  };
};
exports.createNewTermsAlertType = createNewTermsAlertType;