"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPackagePolicyUpdateCallback = exports.getPackagePolicyPostUpdateCallback = exports.getPackagePolicyPostCreateCallback = exports.getPackagePolicyDeleteCallback = exports.getPackagePolicyCreateCallback = exports.getAgentPolicyUpdateCallback = exports.getAgentPolicyPostUpdateCallback = exports.getAgentPolicyCreateCallback = void 0;
var _keys = require("@kbn/security-solution-features/keys");
var _update_deleted_policy_response_actions = require("./handlers/update_deleted_policy_response_actions");
var _create_policy_datastreams = require("./handlers/create_policy_datastreams");
var _update_antivirus_registration_enabled = require("../../common/endpoint/utils/update_antivirus_registration_enabled");
var _validate_policy_against_product_features = require("./handlers/validate_policy_against_product_features");
var _validate_endpoint_package_policy = require("./handlers/validate_endpoint_package_policy");
var _policy_config_helpers = require("../../common/endpoint/models/policy_config_helpers");
var _install_endpoint_security_prebuilt_rule = require("../lib/detection_engine/prebuilt_rules/logic/integrations/install_endpoint_security_prebuilt_rule");
var _create_policy_artifact_manifest = require("./handlers/create_policy_artifact_manifest");
var _create_default_policy = require("./handlers/create_default_policy");
var _validate_policy_against_license = require("./handlers/validate_policy_against_license");
var _validate_integration_config = require("./handlers/validate_integration_config");
var _remove_policy_from_artifacts = require("./handlers/remove_policy_from_artifacts");
var _notify_protection_feature_usage = require("./notify_protection_feature_usage");
var _constants = require("./constants");
var _create_event_filters = require("./handlers/create_event_filters");
var _remove_protection_updates_note = require("./handlers/remove_protection_updates_note");
var _utils = require("../endpoint/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isEndpointPackagePolicy = packagePolicy => {
  var _packagePolicy$packag;
  return ((_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name) === 'endpoint';
};
const getEndpointPolicyForAgentPolicy = async (fleetServices, agentPolicy) => {
  let agentPolicyIntegrations = agentPolicy.package_policies;
  if (!agentPolicyIntegrations) {
    var _fullAgentPolicy$pack;
    const fullAgentPolicy = await fleetServices.agentPolicy.get(fleetServices.savedObjects.createInternalUnscopedSoClient(), agentPolicy.id, true);
    agentPolicyIntegrations = (_fullAgentPolicy$pack = fullAgentPolicy === null || fullAgentPolicy === void 0 ? void 0 : fullAgentPolicy.package_policies) !== null && _fullAgentPolicy$pack !== void 0 ? _fullAgentPolicy$pack : [];
  }
  if (Array.isArray(agentPolicyIntegrations)) {
    for (const integrationPolicy of agentPolicyIntegrations) {
      if (isEndpointPackagePolicy(integrationPolicy)) {
        return integrationPolicy;
      }
    }
  }
  return undefined;
};
const shouldUpdateMetaValues = (endpointPackagePolicy, currentLicenseType, currentCloudInfo, currentClusterName, currentClusterUUID, currentLicenseUUID, currentIsServerlessEnabled) => {
  return endpointPackagePolicy.meta.license !== currentLicenseType || endpointPackagePolicy.meta.cloud !== currentCloudInfo || endpointPackagePolicy.meta.cluster_name !== currentClusterName || endpointPackagePolicy.meta.cluster_uuid !== currentClusterUUID || endpointPackagePolicy.meta.license_uuid !== currentLicenseUUID || endpointPackagePolicy.meta.serverless !== currentIsServerlessEnabled;
};

/**
 * Callback to handle creation of PackagePolicies in Fleet
 */
const getPackagePolicyCreateCallback = (logger, manifestManager, securitySolutionRequestContextFactory, alerts, licenseService, cloud, productFeatures, telemetryConfigProvider) => {
  return async (newPackagePolicy, soClient, esClient, context, request) => {
    var _newPackagePolicy$inp, _newPackagePolicy$inp2, _newPackagePolicy$inp3;
    // callback is called outside request context
    if (!context || !request) {
      logger.debug('PackagePolicyCreateCallback called outside request context. Skipping...');
      return newPackagePolicy;
    }

    // We only care about Endpoint package policies
    if (!isEndpointPackagePolicy(newPackagePolicy)) {
      return newPackagePolicy;
    }
    logger.debug(() => `Checking create of endpoint policy [${newPackagePolicy.id}][${newPackagePolicy.name}] for compliance.`);
    if (newPackagePolicy !== null && newPackagePolicy !== void 0 && newPackagePolicy.inputs) {
      (0, _validate_policy_against_product_features.validatePolicyAgainstProductFeatures)(newPackagePolicy.inputs, productFeatures);
      (0, _validate_endpoint_package_policy.validateEndpointPackagePolicy)(newPackagePolicy.inputs);
    }
    // Optional endpoint integration configuration
    let endpointIntegrationConfig;

    // Check if has endpoint integration configuration input
    const integrationConfigInput = newPackagePolicy === null || newPackagePolicy === void 0 ? void 0 : (_newPackagePolicy$inp = newPackagePolicy.inputs) === null || _newPackagePolicy$inp === void 0 ? void 0 : (_newPackagePolicy$inp2 = _newPackagePolicy$inp.find(input => input.type === _constants.ENDPOINT_INTEGRATION_CONFIG_KEY)) === null || _newPackagePolicy$inp2 === void 0 ? void 0 : (_newPackagePolicy$inp3 = _newPackagePolicy$inp2.config) === null || _newPackagePolicy$inp3 === void 0 ? void 0 : _newPackagePolicy$inp3._config;
    if (integrationConfigInput !== null && integrationConfigInput !== void 0 && integrationConfigInput.value) {
      // The cast below is needed in order to ensure proper typing for the
      // Elastic Defend integration configuration
      endpointIntegrationConfig = integrationConfigInput.value;

      // Validate that the Elastic Defend integration config is valid
      (0, _validate_integration_config.validateIntegrationConfig)(endpointIntegrationConfig, logger);
    }

    // In this callback we are handling an HTTP request to the fleet plugin. Since we use
    // code from the security_solution plugin to handle it (installPrepackagedRules),
    // we need to build the context that is native to security_solution and pass it there.
    const securitySolutionContext = await securitySolutionRequestContextFactory.create(context, request);

    // perform these operations in parallel in order to help in not delaying the API response too much
    const [, manifestValue] = await Promise.all([(0, _install_endpoint_security_prebuilt_rule.installEndpointSecurityPrebuiltRule)({
      logger,
      context: securitySolutionContext,
      request,
      alerts,
      soClient
    }),
    // create the Artifact Manifest for this policy
    (0, _create_policy_artifact_manifest.createPolicyArtifactManifest)(logger, manifestManager)]);
    const esClientInfo = await esClient.info();

    // Add the default endpoint security policy
    const defaultPolicyValue = (0, _create_default_policy.createDefaultPolicy)(licenseService, endpointIntegrationConfig, cloud, esClientInfo, productFeatures, telemetryConfigProvider);
    return {
      // We cast the type here so that any changes to the Endpoint
      // specific data follow the types/schema expected
      ...newPackagePolicy,
      inputs: [{
        type: 'endpoint',
        enabled: true,
        streams: [],
        config: {
          integration_config: endpointIntegrationConfig ? {
            value: endpointIntegrationConfig
          } : {},
          artifact_manifest: {
            value: manifestValue
          },
          policy: {
            value: defaultPolicyValue
          }
        }
      }]
    };
  };
};
exports.getPackagePolicyCreateCallback = getPackagePolicyCreateCallback;
const getPackagePolicyUpdateCallback = (endpointServices, cloud, productFeatures) => {
  const logger = endpointServices.createLogger('endpointPackagePolicyUpdateCallback');
  const licenseService = endpointServices.getLicenseService();
  const featureUsageService = endpointServices.getFeatureUsageService();
  return async (newPackagePolicy, soClient, esClient) => {
    var _endpointIntegrationD, _endpointIntegrationD2, _endpointIntegrationD3, _endpointIntegrationD4, _endpointIntegrationD7, _endpointIntegrationD8, _endpointIntegrationD9, _endpointIntegrationD10;
    if (!isEndpointPackagePolicy(newPackagePolicy)) {
      return newPackagePolicy;
    }
    logger.debug(() => `Checking update of endpoint policy [${newPackagePolicy.id}][${newPackagePolicy.name}] for compliance.`);
    const endpointIntegrationData = newPackagePolicy;

    // Validate that Endpoint Security policy uses only enabled App Features
    (0, _validate_policy_against_product_features.validatePolicyAgainstProductFeatures)(endpointIntegrationData.inputs, productFeatures);

    // Validate that Endpoint Security policy is valid against current license
    if ((_endpointIntegrationD = endpointIntegrationData.inputs) !== null && _endpointIntegrationD !== void 0 && (_endpointIntegrationD2 = _endpointIntegrationD[0]) !== null && _endpointIntegrationD2 !== void 0 && (_endpointIntegrationD3 = _endpointIntegrationD2.config) !== null && _endpointIntegrationD3 !== void 0 && (_endpointIntegrationD4 = _endpointIntegrationD3.policy) !== null && _endpointIntegrationD4 !== void 0 && _endpointIntegrationD4.value) {
      var _endpointIntegrationD5, _endpointIntegrationD6;
      (0, _validate_policy_against_license.validatePolicyAgainstLicense)(// The cast below is needed in order to ensure proper typing for
      // the policy configuration specific for endpoint
      (_endpointIntegrationD5 = endpointIntegrationData.inputs[0].config) === null || _endpointIntegrationD5 === void 0 ? void 0 : (_endpointIntegrationD6 = _endpointIntegrationD5.policy) === null || _endpointIntegrationD6 === void 0 ? void 0 : _endpointIntegrationD6.value, licenseService, logger);
    }

    // Make sure policy includes general expected data
    (0, _validate_endpoint_package_policy.validateEndpointPackagePolicy)(endpointIntegrationData.inputs, 'update');
    if (endpointIntegrationData.id) {
      await (0, _notify_protection_feature_usage.notifyProtectionFeatureUsage)(endpointIntegrationData, await endpointServices.getInternalFleetServices().packagePolicy.get(soClient, endpointIntegrationData.id).catch(_utils.catchAndWrapError), featureUsageService);
    }
    const newEndpointPackagePolicy = (_endpointIntegrationD7 = endpointIntegrationData.inputs[0].config) === null || _endpointIntegrationD7 === void 0 ? void 0 : (_endpointIntegrationD8 = _endpointIntegrationD7.policy) === null || _endpointIntegrationD8 === void 0 ? void 0 : _endpointIntegrationD8.value;
    const esClientInfo = await esClient.info();
    if ((_endpointIntegrationD9 = endpointIntegrationData.inputs[0].config) !== null && _endpointIntegrationD9 !== void 0 && (_endpointIntegrationD10 = _endpointIntegrationD9.policy) !== null && _endpointIntegrationD10 !== void 0 && _endpointIntegrationD10.value && shouldUpdateMetaValues(newEndpointPackagePolicy, licenseService.getLicenseType(), cloud === null || cloud === void 0 ? void 0 : cloud.isCloudEnabled, esClientInfo.cluster_name, esClientInfo.cluster_uuid, licenseService.getLicenseUID(), cloud === null || cloud === void 0 ? void 0 : cloud.isServerlessEnabled)) {
      newEndpointPackagePolicy.meta.license = licenseService.getLicenseType();
      newEndpointPackagePolicy.meta.cloud = cloud === null || cloud === void 0 ? void 0 : cloud.isCloudEnabled;
      newEndpointPackagePolicy.meta.cluster_name = esClientInfo.cluster_name;
      newEndpointPackagePolicy.meta.cluster_uuid = esClientInfo.cluster_uuid;
      newEndpointPackagePolicy.meta.license_uuid = licenseService.getLicenseUID();
      newEndpointPackagePolicy.meta.serverless = cloud === null || cloud === void 0 ? void 0 : cloud.isServerlessEnabled;
      endpointIntegrationData.inputs[0].config.policy.value = newEndpointPackagePolicy;
    }

    // If no Policy Protection allowed (ex. serverless)
    const eventsOnlyPolicy = (0, _policy_config_helpers.isPolicySetToEventCollectionOnly)(newEndpointPackagePolicy);
    if (!productFeatures.isEnabled(_keys.ProductFeatureSecurityKey.endpointPolicyProtections) && !eventsOnlyPolicy.isOnlyCollectingEvents) {
      logger.warn(`Endpoint integration policy [${endpointIntegrationData.id}][${endpointIntegrationData.name}] adjusted due to [endpointPolicyProtections] productFeature not being enabled. Trigger [${eventsOnlyPolicy.message}]`);
      endpointIntegrationData.inputs[0].config.policy.value = (0, _policy_config_helpers.ensureOnlyEventCollectionIsAllowed)(newEndpointPackagePolicy);
    }
    (0, _update_antivirus_registration_enabled.updateAntivirusRegistrationEnabled)(newEndpointPackagePolicy);
    newEndpointPackagePolicy.meta.billable = (0, _policy_config_helpers.isBillablePolicy)(newEndpointPackagePolicy);
    return endpointIntegrationData;
  };
};
exports.getPackagePolicyUpdateCallback = getPackagePolicyUpdateCallback;
const getPackagePolicyPostUpdateCallback = endpointServices => {
  const logger = endpointServices.createLogger('endpointPackagePolicyPostUpdate');
  return async packagePolicy => {
    if (!isEndpointPackagePolicy(packagePolicy)) {
      return packagePolicy;
    }
    logger.debug(`Processing endpoint integration policy (post update): ${packagePolicy.id}`);

    // The check below will run in the background - we don't need to wait for it
    (0, _create_policy_datastreams.createPolicyDataStreamsIfNeeded)({
      endpointServices,
      endpointPolicyIds: [packagePolicy.id]
    }).catch(e => {
      logger.error(`Attempt to check and create DOT datastreams indexes for endpoint integration policy [${packagePolicy.id}] failed`, {
        error: e
      });
    });
    return packagePolicy;
  };
};
exports.getPackagePolicyPostUpdateCallback = getPackagePolicyPostUpdateCallback;
const getPackagePolicyPostCreateCallback = endpointServices => {
  const logger = endpointServices.createLogger('endpointPolicyPostCreate');
  const exceptionsClient = endpointServices.getExceptionListsClient();
  return async packagePolicy => {
    var _packagePolicy$inputs, _packagePolicy$inputs2, _integrationConfig$va;
    // We only care about Endpoint package policies
    if (!exceptionsClient || !isEndpointPackagePolicy(packagePolicy)) {
      return packagePolicy;
    }

    // Check and create internal datastreams for this policy if needed.
    // NOTE: we don't need for it to complete here, thus no `await`.
    (0, _create_policy_datastreams.createPolicyDataStreamsIfNeeded)({
      endpointServices,
      endpointPolicyIds: [packagePolicy.id]
    }).catch(e => {
      logger.error(`Attempt to check and create DOT datastreams indexes for agent policy [${packagePolicy.id}] failed`, {
        error: e
      });
    });
    const integrationConfig = packagePolicy === null || packagePolicy === void 0 ? void 0 : (_packagePolicy$inputs = packagePolicy.inputs[0]) === null || _packagePolicy$inputs === void 0 ? void 0 : (_packagePolicy$inputs2 = _packagePolicy$inputs.config) === null || _packagePolicy$inputs2 === void 0 ? void 0 : _packagePolicy$inputs2.integration_config;
    if (integrationConfig && (integrationConfig === null || integrationConfig === void 0 ? void 0 : (_integrationConfig$va = integrationConfig.value) === null || _integrationConfig$va === void 0 ? void 0 : _integrationConfig$va.eventFilters) !== undefined) {
      (0, _create_event_filters.createEventFilters)(logger, exceptionsClient, integrationConfig.value.eventFilters, packagePolicy).catch(error => {
        logger.error(`Failed to create event filters: ${error}`);
      });
    }
    return packagePolicy;
  };
};
exports.getPackagePolicyPostCreateCallback = getPackagePolicyPostCreateCallback;
const throwAgentTamperProtectionUnavailableError = (logger, policyName, policyId) => {
  const agentTamperProtectionUnavailableError = new Error('Agent Tamper Protection is not allowed in current environment');
  // Agent Policy Service will check for apiPassThrough and rethrow. Route handler will check for statusCode and overwrite.
  agentTamperProtectionUnavailableError.statusCode = 403;
  agentTamperProtectionUnavailableError.apiPassThrough = true;
  logger.error(`Policy [${policyName}:${policyId}] error: Agent Tamper Protection requires Complete Endpoint Security tier`);
  throw agentTamperProtectionUnavailableError;
};
const getAgentPolicyCreateCallback = (logger, productFeatures) => {
  return async agentPolicy => {
    if (agentPolicy.is_protected && !productFeatures.isEnabled(_keys.ProductFeatureSecurityKey.endpointAgentTamperProtection)) {
      throwAgentTamperProtectionUnavailableError(logger, agentPolicy.name, agentPolicy.id);
    }
    return agentPolicy;
  };
};
exports.getAgentPolicyCreateCallback = getAgentPolicyCreateCallback;
const getAgentPolicyUpdateCallback = (logger, productFeatures) => {
  return async agentPolicy => {
    if (agentPolicy.is_protected && !productFeatures.isEnabled(_keys.ProductFeatureSecurityKey.endpointAgentTamperProtection)) {
      throwAgentTamperProtectionUnavailableError(logger, agentPolicy.name, agentPolicy.id);
    }
    return agentPolicy;
  };
};
exports.getAgentPolicyUpdateCallback = getAgentPolicyUpdateCallback;
const getAgentPolicyPostUpdateCallback = endpointServices => {
  const logger = endpointServices.createLogger('endpointPolicyPostUpdate');
  return async agentPolicy => {
    const fleetServices = endpointServices.getInternalFleetServices();
    const endpointPolicy = await getEndpointPolicyForAgentPolicy(fleetServices, agentPolicy);
    if (!endpointPolicy) {
      return agentPolicy;
    }
    logger.debug(`Processing post-update to Fleet agent policy: [${agentPolicy.id}]`);

    // We don't need to `await` for this function to execute. It can be done in the background
    (0, _create_policy_datastreams.createPolicyDataStreamsIfNeeded)({
      endpointServices,
      endpointPolicyIds: [endpointPolicy.id]
    }).catch(e => {
      logger.error(`Attempt to check and create DOT datastreams indexes for agent policy [${endpointPolicy.id}] failed`, {
        error: e
      });
    });
    return agentPolicy;
  };
};
exports.getAgentPolicyPostUpdateCallback = getAgentPolicyPostUpdateCallback;
const getPackagePolicyDeleteCallback = endpointServices => {
  const exceptionsClient = endpointServices.getExceptionListsClient();
  const logger = endpointServices.createLogger('endpointPolicyDeleteCallback');
  return async deletePackagePolicy => {
    const policiesToRemove = [];
    for (const policy of deletePackagePolicy) {
      if (isEndpointPackagePolicy(policy)) {
        logger.debug(`Processing deleted endpoint policy [${policy.id}]`);
        policiesToRemove.push((0, _remove_policy_from_artifacts.removePolicyFromArtifacts)(exceptionsClient, policy, logger));
        policiesToRemove.push((0, _remove_protection_updates_note.removeProtectionUpdatesNote)(endpointServices, policy));
      }
    }

    // Add processing of setting response actions to orphan for integrations (ex. Crowdstrike,
    // SentinelOne, etc) that support response actions
    policiesToRemove.push((0, _update_deleted_policy_response_actions.updateDeletedPolicyResponseActions)(endpointServices, deletePackagePolicy));
    await Promise.all(policiesToRemove);
    logger.debug(`Done processing deleted policies`);
  };
};
exports.getPackagePolicyDeleteCallback = getPackagePolicyDeleteCallback;