"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.inputFieldValue = inputFieldValue;
var _react = require("@testing-library/react");
var _userEvent = _interopRequireDefault(require("@testing-library/user-event"));
var _time_duration = require("@kbn/securitysolution-utils/time_duration");
var _invariant = require("../../../../../../../../common/utils/invariant");
var _to_simple_rule_schedule = require("../../../../../../../../common/api/detection_engine/model/rule_schema/to_simple_rule_schedule");
var _combobox = require("../../../../../../../common/test/eui/combobox");
var _super_select = require("../../../../../../../common/test/eui/super_select");
var _detection_engine = require("../../../../../../../../common/api/detection_engine");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function inputFieldValue(wrapper, params) {
  var _params$value;
  const fieldFinalSide = (0, _react.within)(wrapper).getByTestId(`${params.fieldName}-finalSide`);
  switch (params.fieldName) {
    case 'name':
      await inputText(fieldFinalSide, params.value);
      break;
    case 'description':
      await inputText(fieldFinalSide, params.value);
      break;
    case 'severity':
      await inputSeverity(fieldFinalSide, params.value);
      break;
    case 'severity_mapping':
      await inputSeverityMapping(fieldFinalSide, params.value);
      break;
    case 'risk_score':
      await inputRiskScore(fieldFinalSide, params.value);
      break;
    case 'risk_score_mapping':
      await inputRiskScoreOverride(fieldFinalSide, params.value);
      break;
    case 'references':
      await inputStringsArray(fieldFinalSide, {
        addInputButtonName: 'Add reference URL',
        items: params.value
      });
      break;
    case 'false_positives':
      await inputStringsArray(fieldFinalSide, {
        addInputButtonName: 'Add false positive example',
        items: params.value
      });
      break;
    case 'threat':
      await inputThreat(fieldFinalSide, params.value);
      break;
    case 'note':
      await inputText(fieldFinalSide, params.value);
      break;
    case 'setup':
      await inputText(fieldFinalSide, params.value);
      break;
    case 'related_integrations':
      await inputRelatedIntegrations(fieldFinalSide, params.value);
      break;
    case 'required_fields':
      await inputRequiredFields(fieldFinalSide, params.value);
      break;
    case 'rule_schedule':
      await inputRuleSchedule(fieldFinalSide, params.value);
      break;
    case 'max_signals':
      await inputMaxSignals(fieldFinalSide, params.value);
      break;
    case 'rule_name_override':
      await inputRuleNameOverride(fieldFinalSide, params.value);
      break;
    case 'timestamp_override':
      await inputTimestampOverride(fieldFinalSide, params.value);
      break;
    case 'timeline_template':
      await inputTimelineTemplate(fieldFinalSide, params.value);
      break;
    case 'building_block':
      await inputBuildingBlock(fieldFinalSide, params.value);
      break;
    case 'investigation_fields':
      await inputInvestigationFields(fieldFinalSide, params.value);
      break;
    case 'data_source':
      await inputDataSource(fieldFinalSide, params.value);
      break;
    case 'alert_suppression':
      await inputAlertSuppression(fieldFinalSide, params.value);
      break;
    case 'anomaly_threshold':
      await inputAnomalyThreshold(fieldFinalSide, params.value);
      break;
    case 'kql_query':
      await inputKqlQuery(fieldFinalSide, params.value);
      break;
    case 'eql_query':
      await inputEqlQuery(fieldFinalSide, params.value);
      break;
    case 'esql_query':
      throw new Error('Not implemented');
    case 'history_window_start':
      await inputHistoryWindowStart(fieldFinalSide, params.value);
      break;
    case 'machine_learning_job_id':
      await inputMachineLearningJobId(fieldFinalSide, params.value);
      break;
    case 'new_terms_fields':
      await inputNewTermsFields(fieldFinalSide, params.value);
      break;
    case 'threat_index':
      await inputThreatIndex(fieldFinalSide, params.value);
      break;
    case 'threat_indicator_path':
      await inputText(fieldFinalSide, (_params$value = params.value) !== null && _params$value !== void 0 ? _params$value : '');
      break;
    case 'threat_mapping':
      throw new Error('Not implemented');
    case 'threat_query':
      await inputThreatQuery(fieldFinalSide, params.value);
      break;
    case 'threshold':
      await inputThreshold(fieldFinalSide, params.value);
      break;
  }
}
async function fireEnterEvent(el) {
  await (0, _react.act)(async () => {
    el.focus();
    await _userEvent.default.keyboard('{Enter}');
  });
}
async function inputText(fieldFinalSide, value) {
  await (0, _react.act)(async () => {
    const input = (0, _react.within)(fieldFinalSide).getByRole('textbox');
    _react.fireEvent.change(input, {
      target: {
        value
      }
    });
  });
}
async function inputSeverity(fieldFinalSide, value) {
  const toggleButton = (0, _react.within)(fieldFinalSide).getByTestId('select');
  await (0, _super_select.selectEuiSuperSelectOption)({
    toggleButton,
    optionText: value
  });
}
async function inputSeverityMapping(fieldFinalSide, value) {
  const severityArray = ['low', 'medium', 'high', 'critical'];
  const severityMappingFormRows = (0, _react.within)(fieldFinalSide).getAllByTestId('severityOverrideRow');
  expect(severityMappingFormRows).toHaveLength(severityArray.length);
  for (let i = 0; i < severityArray.length; ++i) {
    const severityLevel = severityArray[i];
    const formRow = severityMappingFormRows[i];
    const [sourceFieldComboboxInput, sourceValueComboboxInput] = (0, _react.within)(formRow).getAllByRole('combobox');
    const mapping = value.find(x => x.severity.toLowerCase() === severityLevel);
    if (mapping) {
      await (0, _react.act)(async () => {
        _react.fireEvent.change(sourceFieldComboboxInput, {
          target: {
            value: mapping.field
          }
        });
      });
      await fireEnterEvent(sourceFieldComboboxInput);
      await (0, _react.act)(async () => {
        _react.fireEvent.change(sourceValueComboboxInput, {
          target: {
            value: mapping.value
          }
        });
      });
      await fireEnterEvent(sourceValueComboboxInput);
    } else {
      // Clear mapping value for the current severity level
      await (0, _react.act)(async () => {
        sourceFieldComboboxInput.focus();
        await _userEvent.default.keyboard('{Backspace}');
      });
    }
  }
}
async function inputRiskScore(fieldFinalSide, value) {
  await (0, _react.act)(async () => {
    // EuiRange is used for Risk Score
    const [riskScoreInput] = (0, _react.within)(fieldFinalSide).getAllByTestId('defaultRiskScore-defaultRiskRange');
    _react.fireEvent.change(riskScoreInput, {
      target: {
        value
      }
    });
  });
}
async function inputRiskScoreOverride(fieldFinalSide, value) {
  (0, _invariant.invariant)(value.length === 1, 'setRiskScoreOverride() expects a single entry risk score mapping');
  const sourceFieldComboboxInput = (0, _react.within)(fieldFinalSide).getByRole('combobox');
  await (0, _react.waitFor)(() => expect(sourceFieldComboboxInput).toBeEnabled(), {
    timeout: 500
  });
  await (0, _react.act)(async () => {
    _react.fireEvent.change(sourceFieldComboboxInput, {
      target: {
        value: value[0].field
      }
    });
  });
  await fireEnterEvent(sourceFieldComboboxInput);
}
async function inputStringsArray(fieldFinalSide, {
  addInputButtonName,
  items
}) {
  await removeExistingItems(fieldFinalSide);
  const addItem = async () => {
    await (0, _react.act)(async () => {
      _react.fireEvent.click((0, _react.within)(fieldFinalSide).getByRole('button', {
        name: addInputButtonName
      }));
    });
  };
  for (let i = 0; i < items.length; ++i) {
    await addItem();
    const inputs = (0, _react.within)(fieldFinalSide).getAllByRole('textbox');
    await (0, _react.act)(async () => {
      _react.fireEvent.change(inputs[i], {
        target: {
          value: items[i]
        }
      });
    });
  }
}

// Limited to tactics
async function inputThreat(fieldFinalSide, value) {
  await removeExistingItems(fieldFinalSide);
  const addTactic = async () => {
    await (0, _react.act)(async () => {
      _react.fireEvent.click((0, _react.within)(fieldFinalSide).getByRole('button', {
        name: 'Add tactic'
      }));
    });
  };
  for (let i = 0; i < value.length; ++i) {
    await addTactic();
    await (0, _super_select.selectEuiSuperSelectOption)({
      toggleButton: (0, _react.within)(fieldFinalSide).getAllByTestId('mitreAttackTactic')[i],
      optionText: `${value[i].tactic.name} (${value[i].tactic.id})`
    });
  }
}

/**
 * Requires mocking response with integrations from `GET /internal/detection_engine/fleet/integrations/all`
 */
async function inputRelatedIntegrations(fieldFinalSide, value) {
  await removeExistingItems(fieldFinalSide, {
    removeButtonName: 'Remove related integration'
  });
  const addIntegration = async () => {
    await (0, _react.act)(async () => {
      _react.fireEvent.click((0, _react.within)(fieldFinalSide).getByRole('button', {
        name: 'Add integration'
      }));
    });
  };
  for (let i = 0; i < value.length; ++i) {
    const {
      package: integrationPackageName,
      version
    } = value[i];
    await addIntegration();
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getAllByTestId('comboBoxToggleListButton')[i],
      // Expect only installed and enabled integrations
      optionText: `${integrationPackageName}Installed: Enabled`
    });
    const packageVersionInput = (0, _react.within)(fieldFinalSide).getAllByRole('textbox')[i];
    await (0, _react.waitFor)(() => expect(packageVersionInput).toBeEnabled(), {
      timeout: 500
    });
    await (0, _react.act)(async () => {
      _react.fireEvent.change(packageVersionInput, {
        target: {
          value: version
        }
      });
    });

    // Make sure the validation has passed
    await (0, _react.waitFor)(() => expect(packageVersionInput).toHaveValue(version), {
      timeout: 500
    });
  }
}
async function inputRequiredFields(fieldFinalSide, value) {
  await removeExistingItems(fieldFinalSide, {
    removeButtonName: 'Remove required field'
  });
  const addRequiredField = async () => {
    await (0, _react.act)(async () => {
      _react.fireEvent.click((0, _react.within)(fieldFinalSide).getByRole('button', {
        name: 'Add required field'
      }));
    });
  };
  for (let i = 0; i < value.length; ++i) {
    const {
      name,
      type
    } = value[i];
    await addRequiredField();
    const formRow = (0, _react.within)(fieldFinalSide).getAllByTestId('requiredFieldsFormRow')[i];
    const [nameInput, typeInput] = (0, _react.within)(formRow).getAllByRole('combobox');
    await (0, _react.act)(async () => {
      _react.fireEvent.change(nameInput, {
        target: {
          value: name
        }
      });
    });
    await fireEnterEvent(nameInput);
    await (0, _react.act)(async () => {
      _react.fireEvent.change(typeInput, {
        target: {
          value: type
        }
      });
    });
    await fireEnterEvent(typeInput);
  }
}
async function inputRuleSchedule(fieldFinalSide, ruleSchedule) {
  const intervalFormRow = (0, _react.within)(fieldFinalSide).getByTestId('intervalFormRow');
  const intervalValueInput = (0, _react.within)(intervalFormRow).getByRole('spinbutton');
  const intervalUnitInput = (0, _react.within)(intervalFormRow).getByRole('combobox');
  const lookBackFormRow = (0, _react.within)(fieldFinalSide).getByTestId('lookbackFormRow');
  const lookBackValueInput = (0, _react.within)(lookBackFormRow).getByRole('spinbutton');
  const lookBackUnitInput = (0, _react.within)(lookBackFormRow).getByRole('combobox');
  const simpleRuleSchedule = (0, _to_simple_rule_schedule.toSimpleRuleSchedule)(ruleSchedule);
  (0, _invariant.invariant)(simpleRuleSchedule, 'Provided rule schedule is not convertible to simple rule schedule');
  const parsedInterval = _time_duration.TimeDuration.parse(simpleRuleSchedule.interval);
  const parsedLookBack = _time_duration.TimeDuration.parse(simpleRuleSchedule.lookback);
  await (0, _react.act)(async () => {
    _react.fireEvent.change(intervalValueInput, {
      target: {
        value: parsedInterval === null || parsedInterval === void 0 ? void 0 : parsedInterval.value
      }
    });
  });
  await (0, _react.act)(async () => {
    _react.fireEvent.change(intervalUnitInput, {
      target: {
        value: parsedInterval === null || parsedInterval === void 0 ? void 0 : parsedInterval.unit
      }
    });
  });
  await (0, _react.act)(async () => {
    _react.fireEvent.change(lookBackValueInput, {
      target: {
        value: parsedLookBack === null || parsedLookBack === void 0 ? void 0 : parsedLookBack.value
      }
    });
  });
  await (0, _react.act)(async () => {
    _react.fireEvent.change(lookBackUnitInput, {
      target: {
        value: parsedLookBack === null || parsedLookBack === void 0 ? void 0 : parsedLookBack.unit
      }
    });
  });
}
async function inputMaxSignals(fieldFinalSide, value) {
  const input = (0, _react.within)(fieldFinalSide).getByRole('spinbutton');
  await (0, _react.act)(async () => {
    _react.fireEvent.change(input, {
      target: {
        value
      }
    });
  });
}
async function inputRuleNameOverride(fieldFinalSide, value) {
  await (0, _react.waitFor)(() => expect((0, _react.within)(fieldFinalSide).getByTestId('comboBoxSearchInput')).toBeEnabled(), {
    timeout: 500
  });
  if (value) {
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxToggleListButton'),
      optionText: value.field_name
    });
  } else {
    await (0, _react.act)(async () => {
      (0, _react.within)(fieldFinalSide).getByTestId('comboBoxSearchInput').focus();
      await _userEvent.default.keyboard('{Backspace}');
    });
  }
}
async function inputTimestampOverride(fieldFinalSide, value) {
  await (0, _react.waitFor)(() => expect((0, _react.within)(fieldFinalSide).getByTestId('comboBoxSearchInput')).toBeEnabled(), {
    timeout: 500
  });
  if (value) {
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxToggleListButton'),
      optionText: value.field_name
    });
  } else {
    await (0, _react.act)(async () => {
      (0, _react.within)(fieldFinalSide).getByTestId('comboBoxSearchInput').focus();
      await _userEvent.default.keyboard('{Backspace}');
    });
  }
}
async function inputTimelineTemplate(fieldFinalSide, value) {
  var _value$timeline_title;
  const timelineSelectToggleButton = (0, _react.within)(fieldFinalSide).getByRole('combobox');
  await (0, _react.act)(async () => {
    _react.fireEvent.click(timelineSelectToggleButton);
  });
  const options = Array.from(document.querySelectorAll('[role="option"]'));
  const lowerCaseOptionText = (_value$timeline_title = value === null || value === void 0 ? void 0 : value.timeline_title.toLocaleLowerCase()) !== null && _value$timeline_title !== void 0 ? _value$timeline_title : 'None';
  const optionToSelect = options.find(option => {
    var _option$textContent;
    return (_option$textContent = option.textContent) === null || _option$textContent === void 0 ? void 0 : _option$textContent.toLowerCase().includes(lowerCaseOptionText);
  });
  if (optionToSelect) {
    await (0, _react.act)(async () => {
      _react.fireEvent.click(optionToSelect);
    });
  } else {
    throw new Error(`Could not find option with text "${lowerCaseOptionText}". Available options: ${options.map(option => option.textContent).join(', ')}`);
  }
}
async function inputBuildingBlock(fieldFinalSide, value) {
  const markGeneratedAlertsAsBuildingBlockAlertsCheckbox = (0, _react.within)(fieldFinalSide).getByRole('checkbox');

  // Field is already in the expected state, exit.
  if (markGeneratedAlertsAsBuildingBlockAlertsCheckbox.checked && value || !markGeneratedAlertsAsBuildingBlockAlertsCheckbox.checked && !value) {
    return;
  }
  await (0, _react.act)(async () => {
    _react.fireEvent.click(markGeneratedAlertsAsBuildingBlockAlertsCheckbox);
  });
}
async function inputInvestigationFields(fieldFinalSide, value) {
  await (0, _react.waitFor)(() => expect((0, _react.within)(fieldFinalSide).queryByTestId('comboBoxClearButton')).toBeVisible());
  await (0, _combobox.clearEuiComboBoxSelection)({
    clearButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxClearButton')
  });
  for (const fieldName of (_value$field_names = value === null || value === void 0 ? void 0 : value.field_names) !== null && _value$field_names !== void 0 ? _value$field_names : []) {
    var _value$field_names;
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxToggleListButton'),
      optionText: fieldName
    });
  }
}
async function inputDataSource(fieldFinalSide, dataSource) {
  if (!dataSource) {
    return;
  }
  switch (dataSource.type) {
    case _detection_engine.DataSourceType.index_patterns:
      await (0, _react.act)(async () => {
        const indexPatternsButton = (0, _react.within)(fieldFinalSide).getByTestId('rule-index-toggle-index_patterns');
        _react.fireEvent.click(indexPatternsButton);
      });
      const indexPatternsEditWrapper = (0, _react.within)(fieldFinalSide).getByTestId('indexPatternEdit');
      await (0, _combobox.clearEuiComboBoxSelection)({
        clearButton: (0, _react.within)(indexPatternsEditWrapper).getByTestId('comboBoxClearButton')
      });
      for (const indexPattern of dataSource.index_patterns) {
        await (0, _combobox.addEuiComboBoxOption)({
          wrapper: indexPatternsEditWrapper,
          optionText: indexPattern
        });
      }
      break;
    case _detection_engine.DataSourceType.data_view:
      await (0, _react.act)(async () => {
        const dataViewButton = (0, _react.within)(fieldFinalSide).getByTestId('rule-index-toggle-data_view');
        _react.fireEvent.click(dataViewButton);
      });
      const dataViewEditWrapper = (0, _react.within)(fieldFinalSide).getByTestId('pick-rule-data-source');
      await (0, _react.waitFor)(() => expect((0, _react.within)(dataViewEditWrapper).queryByTestId('comboBoxToggleListButton')).toBeVisible(), {
        timeout: 500
      });
      await (0, _combobox.clearEuiComboBoxSelection)({
        clearButton: (0, _react.within)(dataViewEditWrapper).getByTestId('comboBoxClearButton')
      });
      if (!dataSource.data_view_id) {
        return;
      }
      await (0, _combobox.selectEuiComboBoxOption)({
        comboBoxToggleButton: (0, _react.within)(dataViewEditWrapper).getByTestId('comboBoxToggleListButton'),
        optionText: dataSource.data_view_id
      });
      break;
  }
}

/**
 * Implements only suppression fields
 */
async function inputAlertSuppression(fieldFinalSide, value) {
  await (0, _combobox.clearEuiComboBoxSelection)({
    clearButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxClearButton')
  });
  if (!value) {
    return;
  }
  for (const fieldName of value.group_by) {
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxToggleListButton'),
      optionText: fieldName
    });
  }
}
async function inputAnomalyThreshold(fieldFinalSide, value) {
  await (0, _react.act)(async () => {
    // EuiRange is used for anomaly threshold
    const [riskScoreInput] = (0, _react.within)(fieldFinalSide).getAllByTestId('anomalyThresholdRange');
    _react.fireEvent.change(riskScoreInput, {
      target: {
        value
      }
    });
  });
}

/**
 * Doesn't support filters and saved queries
 */
async function inputKqlQuery(fieldFinalSide, value) {
  if (value.type !== _detection_engine.KqlQueryType.inline_query) {
    return;
  }
  await (0, _react.waitFor)(() => expect((0, _react.within)(fieldFinalSide).getByRole('textbox')).toBeVisible(), {
    timeout: 500
  });
  await inputText(fieldFinalSide, value.query);
}

/**
 * Doesn't support filters and EQL options
 */
async function inputEqlQuery(fieldFinalSide, value) {
  await (0, _react.waitFor)(() => expect((0, _react.within)(fieldFinalSide).getByRole('textbox')).toBeVisible(), {
    timeout: 500
  });
  await inputText(fieldFinalSide, value.query);
}
async function inputHistoryWindowStart(fieldFinalSide, value) {
  const valueInput = (0, _react.within)(fieldFinalSide).getByTestId('interval');
  const unitInput = (0, _react.within)(fieldFinalSide).getByTestId('timeType');
  (0, _invariant.invariant)(value.startsWith('now-'), 'Unable to parse history window start value');
  const parsed = _time_duration.TimeDuration.parse(value.substring(4));
  (0, _invariant.invariant)(parsed, 'Unable to parse history window start value');
  await (0, _react.act)(async () => {
    _react.fireEvent.change(valueInput, {
      target: {
        value: parsed.value
      }
    });
  });
  await (0, _react.act)(async () => {
    _react.fireEvent.change(unitInput, {
      target: {
        value: parsed.unit
      }
    });
  });
}
async function inputMachineLearningJobId(fieldFinalSide, value) {
  const jobIds = [value].flat();

  // Clear currently selected job
  await (0, _combobox.clearEuiComboBoxSelection)({
    clearButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxClearButton')
  });
  for (const jobId of jobIds) {
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxToggleListButton'),
      optionText: jobId
    });
  }
}
async function inputNewTermsFields(fieldFinalSide, value) {
  await (0, _combobox.clearEuiComboBoxSelection)({
    clearButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxClearButton')
  });
  for (const fieldName of value) {
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxToggleListButton'),
      optionText: fieldName
    });
  }
}
async function inputThreatIndex(fieldFinalSide, value) {
  await (0, _combobox.clearEuiComboBoxSelection)({
    clearButton: (0, _react.within)(fieldFinalSide).getByTestId('comboBoxClearButton')
  });
  for (const indexPattern of value) {
    await (0, _combobox.addEuiComboBoxOption)({
      wrapper: fieldFinalSide,
      optionText: indexPattern
    });
  }
}

/**
 * Doesn't support filters
 */
async function inputThreatQuery(fieldFinalSide, value) {
  await (0, _react.waitFor)(() => expect((0, _react.within)(fieldFinalSide).getByRole('textbox')).toBeVisible(), {
    timeout: 500
  });
  await inputText(fieldFinalSide, value.query);
}
async function inputThreshold(fieldFinalSide, value) {
  const groupByFieldsComboBox = (0, _react.within)(fieldFinalSide).getByTestId('detectionEngineStepDefineRuleThresholdField');
  const thresholdInput = (0, _react.within)(fieldFinalSide).getByTestId('detectionEngineStepDefineRuleThresholdValue');
  await (0, _react.act)(async () => {
    const input = (0, _react.within)(thresholdInput).getByRole('spinbutton');
    _react.fireEvent.change(input, {
      target: {
        value: value.value
      }
    });
  });
  const fields = [value.field].flat();
  await (0, _combobox.clearEuiComboBoxSelection)({
    clearButton: (0, _react.within)(groupByFieldsComboBox).getByTestId('comboBoxClearButton')
  });
  for (const field of fields) {
    await (0, _combobox.selectEuiComboBoxOption)({
      comboBoxToggleButton: (0, _react.within)(groupByFieldsComboBox).getByTestId('comboBoxToggleListButton'),
      optionText: field
    });
  }
}
async function removeExistingItems(wrapper, {
  removeButtonName
} = {
  removeButtonName: 'Delete'
}) {
  const deleteButtons = (0, _react.within)(wrapper).getAllByRole('button', {
    name: removeButtonName
  });
  for (let i = deleteButtons.length - 1; i >= 0; --i) {
    await (0, _react.act)(async () => {
      _react.fireEvent.click(deleteButtons[i]);
    });
  }
}