"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLLMsModels = exports.LLMsQuery = exports.LLMS_QUERY_KEY = void 0;
var _reactQuery = require("@tanstack/react-query");
var _i18n = require("@kbn/i18n");
var _inferenceEndpointUiCommon = require("@kbn/inference-endpoint-ui-common");
var _types = require("../../common/types");
var _models = require("../../common/models");
var _use_kibana = require("./use_kibana");
var _use_load_connectors = require("./use_load_connectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isInferenceActionConnector = connector => {
  return 'config' in connector && 'provider' in connector.config;
};
const mapLlmToModels = {
  [_types.LLMs.openai]: {
    icon: _inferenceEndpointUiCommon.SERVICE_PROVIDERS.openai.icon,
    getModels: (connectorName, includeName) => _models.MODELS.filter(({
      provider
    }) => provider === _types.LLMs.openai).map(model => ({
      label: `${model.name} ${includeName ? `(${connectorName})` : ''}`,
      value: model.model,
      promptTokenLimit: model.promptTokenLimit
    }))
  },
  [_types.LLMs.openai_azure]: {
    icon: _inferenceEndpointUiCommon.SERVICE_PROVIDERS.openai.icon,
    getModels: connectorName => [{
      label: _i18n.i18n.translate('xpack.searchPlayground.openAIAzureModel', {
        defaultMessage: '{name} (Azure OpenAI)',
        values: {
          name: connectorName
        }
      })
    }]
  },
  [_types.LLMs.openai_other]: {
    icon: _inferenceEndpointUiCommon.SERVICE_PROVIDERS.openai.icon,
    getModels: connectorName => [{
      label: _i18n.i18n.translate('xpack.searchPlayground.otherOpenAIModel', {
        defaultMessage: '{name} (OpenAI Compatible Service)',
        values: {
          name: connectorName
        }
      })
    }]
  },
  [_types.LLMs.bedrock]: {
    icon: _inferenceEndpointUiCommon.SERVICE_PROVIDERS.amazonbedrock.icon,
    getModels: () => _models.MODELS.filter(({
      provider
    }) => provider === _types.LLMs.bedrock).map(model => ({
      label: model.name,
      value: model.model,
      promptTokenLimit: model.promptTokenLimit
    }))
  },
  [_types.LLMs.gemini]: {
    icon: _inferenceEndpointUiCommon.SERVICE_PROVIDERS.googlevertexai.icon,
    getModels: () => _models.MODELS.filter(({
      provider
    }) => provider === _types.LLMs.gemini).map(model => ({
      label: model.name,
      value: model.model,
      promptTokenLimit: model.promptTokenLimit
    }))
  },
  [_types.LLMs.inference]: {
    icon: connector => {
      return isInferenceActionConnector(connector) ? _inferenceEndpointUiCommon.SERVICE_PROVIDERS[connector.config.provider].icon : '';
    },
    getModels: (connectorName, _, modelId) => {
      var _MODELS$find;
      return [{
        label: connectorName,
        value: modelId,
        promptTokenLimit: (_MODELS$find = _models.MODELS.find(m => m.model === modelId)) === null || _MODELS$find === void 0 ? void 0 : _MODELS$find.promptTokenLimit
      }];
    }
  }
};
const LLMS_QUERY_KEY = exports.LLMS_QUERY_KEY = ['search-playground', 'llms-models'];
const LLMsQuery = (http, client) => async () => {
  const connectors = await client.fetchQuery({
    queryKey: _use_load_connectors.LOAD_CONNECTORS_QUERY_KEY,
    queryFn: (0, _use_load_connectors.LoadConnectorsQuery)(http),
    retry: false
  });
  const mapConnectorTypeToCount = connectors.reduce((result, connector) => {
    result[connector.type] = (result[connector.type] || 0) + 1;
    return result;
  }, {});
  const models = connectors.reduce((result, connector) => {
    var _connector$config, _connector$config$pro;
    const connectorType = connector.type;
    const llmParams = mapLlmToModels[connectorType];
    if (!llmParams) {
      return result;
    }
    const showConnectorName = Number(mapConnectorTypeToCount === null || mapConnectorTypeToCount === void 0 ? void 0 : mapConnectorTypeToCount[connectorType]) > 1;
    llmParams.getModels(connector.name, false, isInferenceActionConnector(connector) ? (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : (_connector$config$pro = _connector$config.providerConfig) === null || _connector$config$pro === void 0 ? void 0 : _connector$config$pro.model_id : undefined).map(({
      label,
      value,
      promptTokenLimit
    }) => ({
      id: (connector === null || connector === void 0 ? void 0 : connector.id) + label,
      name: label,
      value,
      connectorType: connector.type,
      connectorName: connector.name,
      showConnectorName,
      icon: typeof llmParams.icon === 'function' ? llmParams.icon(connector) : llmParams.icon,
      disabled: !connector,
      connectorId: connector.id,
      promptTokenLimit
    })).forEach(model => result.push(model));
    return result;
  }, []);
  return models;
};
exports.LLMsQuery = LLMsQuery;
const useLLMsModels = () => {
  const client = (0, _reactQuery.useQueryClient)();
  const {
    services: {
      http
    }
  } = (0, _use_kibana.useKibana)();
  const {
    data
  } = (0, _reactQuery.useQuery)(LLMS_QUERY_KEY, LLMsQuery(http, client), {
    keepPreviousData: true,
    retry: false
  });
  return data || [];
};
exports.useLLMsModels = useLLMsModels;