"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createChatService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _onechatCommon = require("@kbn/onechat-common");
var _utils = require("../runner/utils");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Parameters for {@link ChatService.converse}
 */

const createChatService = options => {
  return new ChatServiceImpl(options);
};
exports.createChatService = createChatService;
class ChatServiceImpl {
  constructor({
    inference,
    actions,
    logger,
    conversationService,
    agentService
  }) {
    (0, _defineProperty2.default)(this, "inference", void 0);
    (0, _defineProperty2.default)(this, "actions", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "conversationService", void 0);
    (0, _defineProperty2.default)(this, "agentService", void 0);
    this.inference = inference;
    this.actions = actions;
    this.logger = logger;
    this.conversationService = conversationService;
    this.agentService = agentService;
  }
  converse({
    agentId = _onechatCommon.OneChatDefaultAgentId,
    mode = _onechatCommon.AgentMode.normal,
    conversationId,
    connectorId,
    request,
    nextInput
  }) {
    const isNewConversation = !conversationId;
    return (0, _rxjs.forkJoin)({
      conversationClient: (0, _rxjs.defer)(async () => this.conversationService.getScopedClient({
        request
      })),
      agent: (0, _rxjs.defer)(async () => this.agentService.registry.asPublicRegistry().get({
        agentId,
        request
      })),
      chatModel: (0, _rxjs.defer)(async () => {
        if (connectorId) {
          return connectorId;
        }
        const connectors = await (0, _utils.getConnectorList)({
          actions: this.actions,
          request
        });
        const defaultConnector = (0, _utils.getDefaultConnector)({
          connectors
        });
        return defaultConnector.connectorId;
      }).pipe((0, _rxjs.switchMap)(selectedConnectorId => {
        return this.inference.getChatModel({
          request,
          connectorId: selectedConnectorId,
          chatModelOptions: {}
        });
      }))
    }).pipe((0, _rxjs.switchMap)(({
      conversationClient,
      chatModel,
      agent
    }) => {
      const agentIdentifier = (0, _onechatCommon.toSerializedAgentIdentifier)({
        agentId: agent.agentId,
        providerId: agent.providerId
      });
      const conversation$ = getConversation$({
        agentId: agentIdentifier,
        conversationId,
        conversationClient
      });
      const agentEvents$ = getExecutionEvents$({
        agent,
        mode,
        conversation$,
        nextInput
      });
      const title$ = isNewConversation ? generatedTitle$({
        chatModel,
        conversation$,
        nextInput
      }) : conversation$.pipe((0, _rxjs.switchMap)(conversation => {
        return (0, _rxjs.of)(conversation.title);
      }));
      const roundCompletedEvents$ = agentEvents$.pipe((0, _rxjs.filter)(_onechatCommon.isRoundCompleteEvent));
      const saveOrUpdateAndEmit$ = isNewConversation ? createConversation$({
        agentId: agentIdentifier,
        conversationClient,
        title$,
        roundCompletedEvents$
      }) : updateConversation$({
        conversationClient,
        conversation$,
        title$,
        roundCompletedEvents$
      });
      return (0, _rxjs.merge)(agentEvents$, saveOrUpdateAndEmit$).pipe((0, _rxjs.catchError)(err => {
        this.logger.error(`Error executing agent: ${err.stack}`);
        return (0, _rxjs.throwError)(() => (0, _onechatCommon.isOnechatError)(err) ? err : (0, _onechatCommon.createInternalError)(`Error executing agent: ${err.message}`, {
          statusCode: 500
        }));
      }), (0, _rxjs.shareReplay)());
    }));
  }
}
const generatedTitle$ = ({
  chatModel,
  conversation$,
  nextInput
}) => {
  return conversation$.pipe((0, _rxjs.switchMap)(conversation => {
    return (0, _rxjs.defer)(async () => (0, _utils2.generateConversationTitle)({
      previousRounds: conversation.rounds,
      nextInput,
      chatModel
    })).pipe((0, _rxjs.shareReplay)());
  }));
};

/**
 * Persist a new conversation and emit the corresponding event
 */
const createConversation$ = ({
  agentId,
  conversationClient,
  title$,
  roundCompletedEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    title: title$,
    roundCompletedEvent: roundCompletedEvents$
  }).pipe((0, _rxjs.switchMap)(({
    title,
    roundCompletedEvent
  }) => {
    return conversationClient.create({
      title,
      agentId,
      rounds: [roundCompletedEvent.data.round]
    });
  }), (0, _rxjs.switchMap)(createdConversation => {
    return (0, _rxjs.of)((0, _utils2.createConversationCreatedEvent)(createdConversation));
  }));
};

/**
 * Update an existing conversation and emit the corresponding event
 */
const updateConversation$ = ({
  conversationClient,
  conversation$,
  title$,
  roundCompletedEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    conversation: conversation$,
    title: title$,
    roundCompletedEvent: roundCompletedEvents$
  }).pipe((0, _rxjs.switchMap)(({
    conversation,
    title,
    roundCompletedEvent
  }) => {
    return conversationClient.update({
      id: conversation.id,
      title,
      rounds: [...conversation.rounds, roundCompletedEvent.data.round]
    });
  }), (0, _rxjs.switchMap)(updatedConversation => {
    return (0, _rxjs.of)((0, _utils2.createConversationUpdatedEvent)(updatedConversation));
  }));
};
const getExecutionEvents$ = ({
  conversation$,
  mode,
  nextInput,
  agent
}) => {
  return conversation$.pipe((0, _rxjs.switchMap)(conversation => {
    return new _rxjs.Observable(observer => {
      agent.execute({
        agentParams: {
          agentMode: mode,
          nextInput,
          conversation: conversation.rounds
        },
        onEvent: event => {
          observer.next(event);
        }
      }).then(() => {
        observer.complete();
      }, err => {
        observer.error(err);
      });
      return () => {};
    });
  }), (0, _rxjs.shareReplay)());
};
const getConversation$ = ({
  agentId,
  conversationId,
  conversationClient
}) => {
  return (0, _rxjs.defer)(() => {
    if (conversationId) {
      return conversationClient.get(conversationId);
    } else {
      return (0, _rxjs.of)(placeholderConversation({
        agentId
      }));
    }
  }).pipe((0, _rxjs.shareReplay)());
};
const placeholderConversation = ({
  agentId
}) => {
  return {
    id: (0, _uuid.v4)(),
    title: 'New conversation',
    agentId,
    rounds: [],
    updatedAt: new Date().toISOString(),
    createdAt: new Date().toISOString(),
    user: {
      id: 'unknown',
      username: 'unknown'
    }
  };
};