"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatCompleteRoute = registerChatCompleteRoute;
var _inferenceCommon = require("@kbn/inference-common");
var _sseUtilsServer = require("@kbn/sse-utils-server");
var _schemas = require("./schemas");
var _get_request_aborted_signal = require("./get_request_aborted_signal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerChatCompleteRoute({
  coreSetup,
  router,
  logger
}) {
  async function callChatComplete({
    request,
    stream
  }) {
    const [, pluginsStart, inferenceStart] = await coreSetup.getStartServices();
    const actions = pluginsStart.actions;
    const abortController = new AbortController();
    request.events.aborted$.subscribe(() => abortController.abort());
    const client = inferenceStart.getClient({
      request,
      actions,
      logger
    });
    const {
      connectorId,
      messages,
      system,
      toolChoice,
      tools,
      functionCalling,
      maxRetries,
      modelName,
      retryConfiguration,
      temperature,
      metadata
    } = request.body;
    return client.chatComplete({
      connectorId,
      messages,
      system,
      toolChoice,
      tools,
      functionCalling,
      stream,
      abortSignal: abortController.signal,
      maxRetries,
      modelName,
      retryConfiguration,
      temperature,
      metadata
    });
  }
  router.post({
    path: '/internal/inference/chat_complete',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    },
    validate: {
      body: _schemas.chatCompleteBodySchema
    }
  }, async (context, request, response) => {
    try {
      const chatCompleteResponse = await callChatComplete({
        request,
        stream: false
      });
      return response.ok({
        body: chatCompleteResponse
      });
    } catch (e) {
      var _e$meta$status, _e$meta, _e$code;
      return response.custom({
        statusCode: (0, _inferenceCommon.isInferenceError)(e) ? (_e$meta$status = (_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.status) !== null && _e$meta$status !== void 0 ? _e$meta$status : 500 : 500,
        bypassErrorFormat: true,
        body: {
          type: _inferenceCommon.InferenceTaskEventType.error,
          code: (_e$code = e.code) !== null && _e$code !== void 0 ? _e$code : 'unknown',
          message: e.message
        }
      });
    }
  });
  router.post({
    path: '/internal/inference/chat_complete/stream',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the inference client'
      }
    },
    validate: {
      body: _schemas.chatCompleteBodySchema
    }
  }, async (context, request, response) => {
    const chatCompleteEvents$ = await callChatComplete({
      request,
      stream: true
    });
    return response.ok({
      body: (0, _sseUtilsServer.observableIntoEventSourceStream)(chatCompleteEvents$, {
        logger,
        signal: (0, _get_request_aborted_signal.getRequestAbortedSignal)(request)
      })
    });
  });
}