"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDefaultSettings = createDefaultSettings;
exports.getSettings = getSettings;
exports.getSettingsOrUndefined = getSettingsOrUndefined;
exports.saveSettings = saveSettings;
exports.settingsSetup = settingsSetup;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _constants = require("../../common/constants");
var _errors = require("../errors");
var _app_context = require("./app_context");
var _audit_logging = require("./audit_logging");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function mapSettingsSO(settingsSo) {
  var _settingsSo$attribute, _settingsSo$attribute2;
  return {
    id: settingsSo.id,
    version: settingsSo.version,
    secret_storage_requirements_met: settingsSo.attributes.secret_storage_requirements_met,
    output_secret_storage_requirements_met: settingsSo.attributes.output_secret_storage_requirements_met,
    has_seen_add_data_notice: settingsSo.attributes.has_seen_add_data_notice,
    prerelease_integrations_enabled: settingsSo.attributes.prerelease_integrations_enabled,
    use_space_awareness_migration_status: (_settingsSo$attribute = settingsSo.attributes.use_space_awareness_migration_status) !== null && _settingsSo$attribute !== void 0 ? _settingsSo$attribute : undefined,
    use_space_awareness_migration_started_at: (_settingsSo$attribute2 = settingsSo.attributes.use_space_awareness_migration_started_at) !== null && _settingsSo$attribute2 !== void 0 ? _settingsSo$attribute2 : undefined,
    preconfigured_fields: getConfigFleetServerHosts() ? ['fleet_server_hosts'] : [],
    delete_unenrolled_agents: settingsSo.attributes.delete_unenrolled_agents
  };
}
async function getSettings(soClient) {
  const res = await soClient.find({
    type: _constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE
  });
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'get',
    id: _constants.GLOBAL_SETTINGS_ID,
    savedObjectType: _constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE
  });
  if (res.total === 0) {
    throw _boom.default.notFound('Global settings not found');
  }
  return mapSettingsSO(res.saved_objects[0]);
}
async function getSettingsOrUndefined(soClient) {
  try {
    return await getSettings(soClient);
  } catch (e) {
    if (e.isBoom && e.output.statusCode === 404) {
      return undefined;
    }
    throw e;
  }
}
async function settingsSetup(soClient) {
  try {
    const config = _app_context.appContextService.getConfig();
    const settings = await getSettings(soClient);
    if (config !== null && config !== void 0 && config.prereleaseEnabledByDefault && !settings.prerelease_integrations_enabled) {
      await saveSettings(soClient, {
        prerelease_integrations_enabled: config === null || config === void 0 ? void 0 : config.prereleaseEnabledByDefault
      });
    }
  } catch (e) {
    if (e.isBoom && e.output.statusCode === 404) {
      const defaultSettings = createDefaultSettings();
      return saveSettings(soClient, defaultSettings);
    }
    throw e;
  }
}
async function saveSettings(soClient, newData, options) {
  const data = (0, _lodash.omit)({
    ...newData
  }, 'fleet_server_hosts');
  const {
    createWithOverwrite,
    ...updateOptions
  } = options !== null && options !== void 0 ? options : {};
  try {
    var _settings$delete_unen;
    const settings = await getSettings(soClient);
    if (!(options !== null && options !== void 0 && options.fromSetup) && (_settings$delete_unen = settings.delete_unenrolled_agents) !== null && _settings$delete_unen !== void 0 && _settings$delete_unen.is_preconfigured && data.delete_unenrolled_agents) {
      throw new _errors.DeleteUnenrolledAgentsPreconfiguredError(`Setting delete_unenrolled_agents is preconfigured as 'enableDeleteUnenrolledAgents' and cannot be updated outside of kibana config file.`);
    }
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id: settings.id,
      savedObjectType: _constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE
    });
    const res = await soClient.update(_constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE, settings.id, data, updateOptions);
    return {
      id: settings.id,
      ...res.attributes
    };
  } catch (e) {
    if (e.isBoom && e.output.statusCode === 404) {
      const defaultSettings = createDefaultSettings();
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'create',
        id: _constants.GLOBAL_SETTINGS_ID,
        savedObjectType: _constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE
      });
      const res = await soClient.create(_constants.GLOBAL_SETTINGS_SAVED_OBJECT_TYPE, {
        ...defaultSettings,
        ...data
      }, {
        id: _constants.GLOBAL_SETTINGS_ID,
        // Do not overwrite if version is passed
        overwrite: typeof createWithOverwrite === 'undefined' ? true : createWithOverwrite
      });
      return {
        id: res.id,
        ...res.attributes
      };
    }
    throw e;
  }
}
function getConfigFleetServerHosts() {
  var _config$agents, _config$agents$fleet_, _config$agents2, _config$agents2$fleet;
  const config = _app_context.appContextService.getConfig();
  return config !== null && config !== void 0 && (_config$agents = config.agents) !== null && _config$agents !== void 0 && (_config$agents$fleet_ = _config$agents.fleet_server) !== null && _config$agents$fleet_ !== void 0 && _config$agents$fleet_.hosts && config.agents.fleet_server.hosts.length > 0 ? config === null || config === void 0 ? void 0 : (_config$agents2 = config.agents) === null || _config$agents2 === void 0 ? void 0 : (_config$agents2$fleet = _config$agents2.fleet_server) === null || _config$agents2$fleet === void 0 ? void 0 : _config$agents2$fleet.hosts : undefined;
}
function createDefaultSettings() {
  const config = _app_context.appContextService.getConfig();
  const settings = {
    prerelease_integrations_enabled: !!(config !== null && config !== void 0 && config.prereleaseEnabledByDefault)
  };
  if (_app_context.appContextService.getExperimentalFeatures().useSpaceAwareness) {
    settings.use_space_awareness_migration_status = 'success';
  }
  return settings;
}