"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFieldsByTypeRetriever = getFieldsByTypeRetriever;
exports.suggest = suggest;
var _lodash = require("lodash");
var _esqlAst = require("@kbn/esql-ast");
var _esqlTypes = require("@kbn/esql-types");
var _esql_types = require("../shared/esql_types");
var _helpers = require("../shared/helpers");
var _constants = require("../shared/constants");
var _user_defined_columns = require("../shared/user_defined_columns");
var _complete_items = require("./complete_items");
var _factories = require("./factories");
var _context = require("../shared/context");
var _resources_helpers = require("../shared/resources_helpers");
var _helper = require("./helper");
var _types = require("../definitions/types");
var _all_operators = require("../definitions/all_operators");
var _suggestions = require("./recommended_queries/suggestions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function suggest(fullText, offset, context, resourceRetriever) {
  var _resourceRetriever$ca, _resourceRetriever$ca2;
  // Partition out to inner ast / ast context for the latest command
  const innerText = fullText.substring(0, offset);
  const correctedQuery = (0, _helper.correctQuerySyntax)(innerText);
  const {
    ast,
    root
  } = (0, _esqlAst.parse)(correctedQuery, {
    withFormatting: true
  });
  const astContext = (0, _context.getAstContext)(innerText, ast, offset);
  if (astContext.type === 'comment') {
    return [];
  }

  // build the correct query to fetch the list of fields
  const queryForFields = (0, _helper.getQueryForFields)(correctedQuery, root);
  const {
    getFieldsByType,
    getFieldsMap
  } = getFieldsByTypeRetriever(queryForFields.replace(_constants.EDITOR_MARKER, ''), resourceRetriever, innerText);
  const supportsControls = (_resourceRetriever$ca = resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ca2 = resourceRetriever.canSuggestVariables) === null || _resourceRetriever$ca2 === void 0 ? void 0 : _resourceRetriever$ca2.call(resourceRetriever)) !== null && _resourceRetriever$ca !== void 0 ? _resourceRetriever$ca : false;
  const getVariables = resourceRetriever === null || resourceRetriever === void 0 ? void 0 : resourceRetriever.getVariables;
  const getSources = (0, _resources_helpers.getSourcesHelper)(resourceRetriever);
  const {
    getPolicies,
    getPolicyMetadata
  } = getPolicyRetriever(resourceRetriever);
  if (astContext.type === 'newCommand') {
    // propose main commands here
    // resolve particular commands suggestions after
    // filter source commands if already defined
    let suggestions = (0, _complete_items.getCommandAutocompleteDefinitions)((0, _helpers.getAllCommands)());
    if (!ast.length) {
      // Display the recommended queries if there are no commands (empty state)
      const recommendedQueriesSuggestions = [];
      if (getSources) {
        var _await$resourceRetrie, _resourceRetriever$ge;
        let fromCommand = '';
        const sources = await getSources();
        const visibleSources = sources.filter(source => !source.hidden);
        if (visibleSources.find(source => source.name.startsWith('logs'))) {
          fromCommand = 'FROM logs*';
        } else if (visibleSources.length) {
          fromCommand = `FROM ${visibleSources[0].name}`;
        }
        const {
          getFieldsByType: getFieldsByTypeEmptyState
        } = getFieldsByTypeRetriever(fromCommand, resourceRetriever, innerText);
        const editorExtensions = (_await$resourceRetrie = await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge = resourceRetriever.getEditorExtensions) === null || _resourceRetriever$ge === void 0 ? void 0 : _resourceRetriever$ge.call(resourceRetriever, fromCommand))) !== null && _await$resourceRetrie !== void 0 ? _await$resourceRetrie : {
          recommendedQueries: []
        };
        const recommendedQueriesSuggestionsFromExtensions = (0, _suggestions.mapRecommendedQueriesFromExtensions)(editorExtensions.recommendedQueries);
        const recommendedQueriesSuggestionsFromStaticTemplates = await (0, _suggestions.getRecommendedQueriesSuggestionsFromStaticTemplates)(getFieldsByTypeEmptyState, fromCommand);
        recommendedQueriesSuggestions.push(...recommendedQueriesSuggestionsFromExtensions, ...recommendedQueriesSuggestionsFromStaticTemplates);
      }
      const sourceCommandsSuggestions = suggestions.filter(_helpers.isSourceCommand);
      return [...sourceCommandsSuggestions, ...recommendedQueriesSuggestions];
    }

    // If the last command is not a FORK, RRF should not be suggested.
    const lastCommand = root.commands[root.commands.length - 1];
    if (lastCommand.name !== 'fork') {
      suggestions = suggestions.filter(def => def.label !== 'RRF');
    }
    return suggestions.filter(def => !(0, _helpers.isSourceCommand)(def));
  }

  // ToDo: Reconsider where it belongs when this is resolved https://github.com/elastic/kibana/issues/216492
  const lastCharacterTyped = innerText[innerText.length - 1];
  let controlSuggestions = [];
  if (lastCharacterTyped === _constants.ESQL_VARIABLES_PREFIX) {
    controlSuggestions = (0, _factories.getControlSuggestionIfSupported)(Boolean(supportsControls), _esqlTypes.ESQLVariableType.VALUES, getVariables, false);
    return controlSuggestions;
  }
  if (astContext.type === 'expression') {
    const commandsSpecificSuggestions = await getSuggestionsWithinCommandExpression(innerText, ast, astContext, getSources, getFieldsByType, getFieldsMap, getPolicies, getPolicyMetadata, getVariables, resourceRetriever === null || resourceRetriever === void 0 ? void 0 : resourceRetriever.getPreferences, resourceRetriever, supportsControls);
    return commandsSpecificSuggestions;
  }
  if (astContext.type === 'function') {
    const getCommandAndOptionWithinFORK = command => {
      var _subCommand;
      let option;
      let subCommand;
      _esqlAst.Walker.walk(command, {
        visitCommandOption: _node => {
          option = _node;
        },
        visitCommand: _node => {
          subCommand = _node;
        }
      });
      return {
        option,
        command: (_subCommand = subCommand) !== null && _subCommand !== void 0 ? _subCommand : command
      };
    };
    const functionsSpecificSuggestions = await getFunctionArgsSuggestions(innerText, ast, {
      ...astContext,
      ...(astContext.command.name === 'fork' ? getCommandAndOptionWithinFORK(astContext.command) : {})
    }, getFieldsByType, getFieldsMap, fullText, offset, getVariables, supportsControls);
    return functionsSpecificSuggestions;
  }
  if (astContext.type === 'list') {
    return getListArgsSuggestions(innerText, ast, astContext, getFieldsByType, getFieldsMap, getPolicyMetadata);
  }
  return [];
}
function getFieldsByTypeRetriever(queryForFields, resourceRetriever, fullQuery) {
  var _resourceRetriever$ca3, _resourceRetriever$ca4;
  const helpers = (0, _resources_helpers.getFieldsByTypeHelper)(queryForFields, resourceRetriever);
  const getVariables = resourceRetriever === null || resourceRetriever === void 0 ? void 0 : resourceRetriever.getVariables;
  const canSuggestVariables = (_resourceRetriever$ca3 = resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ca4 = resourceRetriever.canSuggestVariables) === null || _resourceRetriever$ca4 === void 0 ? void 0 : _resourceRetriever$ca4.call(resourceRetriever)) !== null && _resourceRetriever$ca3 !== void 0 ? _resourceRetriever$ca3 : false;
  const queryString = fullQuery !== null && fullQuery !== void 0 ? fullQuery : queryForFields;
  const lastCharacterTyped = queryString[queryString.length - 1];
  const lastCharIsQuestionMark = lastCharacterTyped === _constants.ESQL_VARIABLES_PREFIX;
  return {
    getFieldsByType: async (expectedType = 'any', ignored = [], options) => {
      var _await$resourceRetrie2, _resourceRetriever$ge2;
      const updatedOptions = {
        ...options,
        supportsControls: canSuggestVariables && !lastCharIsQuestionMark
      };
      const editorExtensions = (_await$resourceRetrie2 = await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge2 = resourceRetriever.getEditorExtensions) === null || _resourceRetriever$ge2 === void 0 ? void 0 : _resourceRetriever$ge2.call(resourceRetriever, queryForFields))) !== null && _await$resourceRetrie2 !== void 0 ? _await$resourceRetrie2 : {
        recommendedQueries: [],
        recommendedFields: []
      };
      const recommendedFieldsFromExtensions = editorExtensions.recommendedFields;
      const fields = await helpers.getFieldsByType(expectedType, ignored);
      return (0, _factories.buildFieldsDefinitionsWithMetadata)(fields, recommendedFieldsFromExtensions, updatedOptions, getVariables);
    },
    getFieldsMap: helpers.getFieldsMap
  };
}
function getPolicyRetriever(resourceRetriever) {
  const helpers = (0, _resources_helpers.getPolicyHelper)(resourceRetriever);
  return {
    getPolicies: async () => {
      const policies = await helpers.getPolicies();
      return (0, _factories.buildPoliciesDefinitions)(policies);
    },
    getPolicyMetadata: helpers.getPolicyMetadata
  };
}
function findNewUserDefinedColumn(userDefinedColumns) {
  let autoGeneratedColumnCounter = 0;
  let name = `col${autoGeneratedColumnCounter++}`;
  while (userDefinedColumns.has(name)) {
    name = `col${autoGeneratedColumnCounter++}`;
  }
  return name;
}
async function getSuggestionsWithinCommandExpression(innerText, commands, astContext, getSources, getColumnsByType, getFieldsMap, getPolicies, getPolicyMetadata, getVariables, getPreferences, callbacks, supportsControls) {
  var _astContext$containin, _astContext$node, _astContext$node2;
  const commandDef = (0, _helpers.getCommandDefinition)(astContext.command.name);

  // collect all fields + userDefinedColumns to suggest
  const fieldsMap = await getFieldsMap();
  const anyUserDefinedColumns = (0, _user_defined_columns.collectUserDefinedColumns)(commands, fieldsMap, innerText);
  const references = {
    fields: fieldsMap,
    userDefinedColumns: anyUserDefinedColumns
  };

  // For now, we don't suggest for expressions within any function besides CASE
  // e.g. CASE(field != /)
  //
  // So, it is handled as a special branch...
  if (((_astContext$containin = astContext.containingFunction) === null || _astContext$containin === void 0 ? void 0 : _astContext$containin.name) === 'case' && !Array.isArray(astContext.node) && ((_astContext$node = astContext.node) === null || _astContext$node === void 0 ? void 0 : _astContext$node.type) === 'function' && ((_astContext$node2 = astContext.node) === null || _astContext$node2 === void 0 ? void 0 : _astContext$node2.subtype) === 'binary-expression') {
    return await (0, _helper.getSuggestionsToRightOfOperatorExpression)({
      queryText: innerText,
      location: (0, _types.getLocationFromCommandOrOptionName)(astContext.command.name),
      rootOperator: astContext.node,
      getExpressionType: expression => (0, _helpers.getExpressionType)(expression, references.fields, references.userDefinedColumns),
      getColumnsByType
    });
  }

  // Function returning suggestions from static templates and editor extensions
  const getRecommendedQueries = async (queryString, prefix = '') => {
    var _await$callbacks$getE, _callbacks$getEditorE;
    const editorExtensions = (_await$callbacks$getE = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getEditorE = callbacks.getEditorExtensions) === null || _callbacks$getEditorE === void 0 ? void 0 : _callbacks$getEditorE.call(callbacks, queryString))) !== null && _await$callbacks$getE !== void 0 ? _await$callbacks$getE : {
      recommendedQueries: []
    };
    const recommendedQueriesFromExtensions = (0, _suggestions.getRecommendedQueriesTemplatesFromExtensions)(editorExtensions.recommendedQueries);
    const recommendedQueriesFromTemplates = await (0, _suggestions.getRecommendedQueriesSuggestionsFromStaticTemplates)(getColumnsByType, prefix);
    return [...recommendedQueriesFromExtensions, ...recommendedQueriesFromTemplates];
  };
  return commandDef.suggest({
    innerText,
    command: astContext.command,
    getColumnsByType,
    getAllColumnNames: () => Array.from(fieldsMap.keys()),
    columnExists: col => Boolean((0, _helpers.getColumnByName)(col, references)),
    getSuggestedUserDefinedColumnName: extraFieldNames => {
      if (!(extraFieldNames !== null && extraFieldNames !== void 0 && extraFieldNames.length)) {
        return findNewUserDefinedColumn(anyUserDefinedColumns);
      }
      const augmentedFieldsMap = new Map(fieldsMap);
      extraFieldNames.forEach(name => {
        augmentedFieldsMap.set(name, {
          name,
          type: 'double'
        });
      });
      return findNewUserDefinedColumn((0, _user_defined_columns.collectUserDefinedColumns)(commands, augmentedFieldsMap, innerText));
    },
    getExpressionType: expression => (0, _helpers.getExpressionType)(expression, references.fields, references.userDefinedColumns),
    getPreferences,
    definition: commandDef,
    getSources,
    getRecommendedQueriesSuggestions: (queryString, prefix) => getRecommendedQueries(queryString, prefix),
    getSourcesFromQuery: type => (0, _helper.getSourcesFromCommands)(commands, type),
    previousCommands: commands,
    callbacks,
    getVariables,
    supportsControls,
    getPolicies,
    getPolicyMetadata,
    references
  });
}
const addCommaIf = (condition, text) => condition ? `${text},` : text;
async function getFunctionArgsSuggestions(innerText, commands, {
  command,
  option,
  node
}, getFieldsByType, getFieldsMap, fullText, offset, getVariables, supportsControls) {
  const fnDefinition = (0, _helpers.getFunctionDefinition)(node.name);
  // early exit on no hit
  if (!fnDefinition) {
    return [];
  }
  const fieldsMap = await getFieldsMap();
  const anyUserDefinedColumns = (0, _user_defined_columns.collectUserDefinedColumns)(commands, fieldsMap, innerText);
  const references = {
    fields: fieldsMap,
    userDefinedColumns: anyUserDefinedColumns
  };
  const userDefinedColumnsExcludingCurrentCommandOnes = (0, _user_defined_columns.excludeUserDefinedColumnsFromCurrentCommand)(commands, command, fieldsMap, innerText);
  const {
    typesToSuggestNext,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex
  } = (0, _helper.getValidSignaturesAndTypesToSuggestNext)(node, references, fnDefinition, fullText, offset);
  const arg = enrichedArgs[argIndex];

  // Whether to prepend comma to suggestion string
  // E.g. if true, "fieldName" -> "fieldName, "
  const isCursorFollowedByComma = fullText ? fullText.slice(offset, fullText.length).trimStart().startsWith(',') : false;
  const canBeBooleanCondition =
  // For `CASE()`, there can be multiple conditions, so keep suggesting fields and functions if possible
  fnDefinition.name === 'case' ||
  // If the type is explicitly a boolean condition
  typesToSuggestNext.some(t => t && t.type === 'boolean' && t.name === 'condition');
  const shouldAddComma = hasMoreMandatoryArgs && fnDefinition.type !== _types.FunctionDefinitionTypes.OPERATOR && !isCursorFollowedByComma && !canBeBooleanCondition;
  const shouldAdvanceCursor = hasMoreMandatoryArgs && fnDefinition.type !== _types.FunctionDefinitionTypes.OPERATOR && !isCursorFollowedByComma;
  const suggestedConstants = (0, _lodash.uniq)(typesToSuggestNext.map(d => d.literalSuggestions || d.acceptedValues).filter(d => d).flat());
  if (suggestedConstants.length) {
    return (0, _factories.buildValueDefinitions)(suggestedConstants, {
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    });
  }
  const suggestions = [];
  const noArgDefined = !arg;
  const isUnknownColumn = arg && (0, _helpers.isColumnItem)(arg) && !(0, _helpers.getColumnExists)(arg, {
    fields: fieldsMap,
    userDefinedColumns: userDefinedColumnsExcludingCurrentCommandOnes
  });
  if (noArgDefined || isUnknownColumn) {
    // ... | EVAL fn( <suggest>)
    // ... | EVAL fn( field, <suggest>)

    const commandArgIndex = command.args.findIndex(cmdArg => (0, _helpers.isSingleItem)(cmdArg) && cmdArg.location.max >= node.location.max);
    const finalCommandArgIndex = command.name !== 'stats' ? -1 : commandArgIndex < 0 ? Math.max(command.args.length - 1, 0) : commandArgIndex;
    const finalCommandArg = command.args[finalCommandArgIndex];
    const fnToIgnore = [];
    if (node.subtype === 'variadic-call') {
      // for now, this getFunctionArgsSuggestions is being used in STATS to suggest for
      // operators. When that is fixed, we can remove this "is variadic-call" check
      // and always exclude the grouping functions
      fnToIgnore.push(...(0, _helpers.getAllFunctions)({
        type: _types.FunctionDefinitionTypes.GROUPING
      }).map(({
        name
      }) => name));
    }
    if (command.name !== 'stats' || (0, _helpers.isOptionItem)(finalCommandArg) && finalCommandArg.name === 'by') {
      // ignore the current function
      fnToIgnore.push(node.name);
    } else {
      fnToIgnore.push(...(0, _helper.getFunctionsToIgnoreForStats)(command, finalCommandArgIndex), ...((0, _helper.isAggFunctionUsedAlready)(command, finalCommandArgIndex) ? (0, _helpers.getAllFunctions)({
        type: _types.FunctionDefinitionTypes.AGG
      }).map(({
        name
      }) => name) : []));
    }
    // Separate the param definitions into two groups:
    // fields should only be suggested if the param isn't constant-only,
    // and constant suggestions should only be given if it is.
    //
    // TODO - consider incorporating the literalOptions into this
    //
    // TODO — improve this to inherit the constant flag from the outer function
    // (e.g. if func1's first parameter is constant-only, any nested functions should
    // inherit that constraint: func1(func2(shouldBeConstantOnly)))
    //
    const constantOnlyParamDefs = typesToSuggestNext.filter(p => p.constantOnly || /_duration/.test(p.type));
    const getTypesFromParamDefs = paramDefs => {
      return Array.from(new Set(paramDefs.map(({
        type
      }) => type)));
    };

    // Literals
    suggestions.push(...(0, _factories.getCompatibleLiterals)(getTypesFromParamDefs(constantOnlyParamDefs), {
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs,
      supportsControls
    }, getVariables));
    const ensureKeywordAndText = types => {
      if (types.includes('keyword') && !types.includes('text')) {
        types.push('text');
      }
      if (types.includes('text') && !types.includes('keyword')) {
        types.push('keyword');
      }
      return types;
    };

    // Fields

    suggestions.push(...(0, _helper.pushItUpInTheList)(await getFieldsByType(
    // For example, in case() where we are expecting a boolean condition
    // we can accept any field types (field1 !== field2)
    canBeBooleanCondition ? ['any'] :
    // @TODO: have a way to better suggest constant only params
    ensureKeywordAndText(getTypesFromParamDefs(typesToSuggestNext.filter(d => !d.constantOnly))), [], {
      addComma: shouldAddComma,
      advanceCursor: shouldAdvanceCursor,
      openSuggestions: shouldAdvanceCursor
    }), true));

    // Functions
    if (typesToSuggestNext.every(d => !d.fieldsOnly)) {
      var _option$name;
      suggestions.push(...(0, _factories.getFunctionSuggestions)({
        location: (0, _types.getLocationFromCommandOrOptionName)((_option$name = option === null || option === void 0 ? void 0 : option.name) !== null && _option$name !== void 0 ? _option$name : command.name),
        returnTypes: canBeBooleanCondition ? ['any'] : ensureKeywordAndText(getTypesFromParamDefs(typesToSuggestNext)),
        ignored: fnToIgnore
      }).map(suggestion => ({
        ...suggestion,
        text: addCommaIf(shouldAddComma, suggestion.text)
      })));
    }

    // could also be in stats (bucket) but our autocomplete is not great yet
    if (getTypesFromParamDefs(typesToSuggestNext).includes('date') && ['where', 'eval'].includes(command.name) && !_constants.FULL_TEXT_SEARCH_FUNCTIONS.includes(fnDefinition.name) || command.name === 'stats' && typesToSuggestNext.some(t => t && t.type === 'date' && t.constantOnly === true)) suggestions.push(...(0, _factories.getDateLiterals)({
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    }));
  }

  // for eval and row commands try also to complete numeric literals with time intervals where possible
  if (arg) {
    if (command.name !== 'stats') {
      if ((0, _helpers.isLiteralItem)(arg) && (0, _esql_types.isNumericType)(arg.literalType)) {
        // ... | EVAL fn(2 <suggest>)
        suggestions.push(...(0, _factories.getCompatibleLiterals)(['time_literal_unit'], {
          addComma: shouldAddComma,
          advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
        }));
      }
    }
    // Suggest comparison functions for boolean conditions
    if (canBeBooleanCondition) {
      suggestions.push(..._all_operators.comparisonFunctions.map(({
        name,
        description
      }) => ({
        label: name,
        text: name,
        kind: 'Function',
        detail: description
      })));
    }
    if (hasMoreMandatoryArgs) {
      // Suggest a comma if there's another argument for the function
      suggestions.push(_complete_items.commaCompleteItem);
    }
  }

  // For special case of COUNT, suggest * if cursor is in empty spot
  // e.g. count( / ) -> suggest `*`
  if (fnDefinition.name === 'count' && !arg) {
    suggestions.push(_complete_items.allStarConstant);
  }
  return suggestions;
}
async function getListArgsSuggestions(innerText, commands, {
  command,
  node
}, getFieldsByType, getFieldsMaps, getPolicyMetadata) {
  const suggestions = [];

  // node is supposed to be the function who support a list argument (like the "in" operator)
  // so extract the type of the first argument and suggest fields of that type
  if (node && (0, _helpers.isFunctionItem)(node)) {
    const list = node === null || node === void 0 ? void 0 : node.args[1];
    if ((0, _esqlAst.isList)(list)) {
      const noParens = list.location.min === 0 && list.location.max === 0;
      if (noParens) {
        suggestions.push(_complete_items.listCompleteItem);
        return suggestions;
      }
    }
    const fieldsMap = await getFieldsMaps();
    const anyUserDefinedColumns = (0, _user_defined_columns.collectUserDefinedColumns)(commands, fieldsMap, innerText);
    // extract the current node from the userDefinedColumns inferred
    anyUserDefinedColumns.forEach((values, key) => {
      if (values.some(v => v.location === node.location)) {
        anyUserDefinedColumns.delete(key);
      }
    });
    const [firstArg] = node.args;
    if ((0, _helpers.isColumnItem)(firstArg)) {
      const argType = (0, _helper.extractTypeFromASTArg)(firstArg, {
        fields: fieldsMap,
        userDefinedColumns: anyUserDefinedColumns
      });
      if (argType) {
        // do not propose existing columns again
        const otherArgs = (0, _esqlAst.isList)(list) ? list.values : node.args.filter(Array.isArray).flat().filter(_helpers.isColumnItem);
        suggestions.push(...(await (0, _helper.getFieldsOrFunctionsSuggestions)([argType], (0, _types.getLocationFromCommandOrOptionName)(command.name), getFieldsByType, {
          functions: true,
          fields: true,
          userDefinedColumns: anyUserDefinedColumns
        }, {
          ignoreColumns: [firstArg.name, ...otherArgs.map(({
            name
          }) => name)]
        })));
      }
    }
  }
  return suggestions;
}