"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerHasEsDataRoute = exports.patterns = exports.crossClusterPatterns = exports.createHandler = void 0;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const patterns = exports.patterns = ['*', '-.*'].concat(_common.DEFAULT_ASSETS_TO_IGNORE.DATA_STREAMS_TO_IGNORE.map(ds => `-${ds}`));
const crossClusterPatterns = exports.crossClusterPatterns = patterns.map(ds => `*:${ds}`);
const createHandler = (parentLogger, hasEsDataTimeout) => async (ctx, _, res) => {
  const logger = parentLogger.get('hasEsData');
  const core = await ctx.core;
  const elasticsearchClient = core.elasticsearch.client.asCurrentUser;
  const commonParams = {
    elasticsearchClient,
    logger,
    res,
    hasEsDataTimeout
  };
  const localDataResponse = await hasEsData({
    ...commonParams,
    matchPatterns: patterns,
    timeoutReason: _common.HasEsDataFailureReason.localDataTimeout
  });
  if (localDataResponse) {
    return localDataResponse;
  }
  const remoteDataResponse = await hasEsData({
    ...commonParams,
    matchPatterns: crossClusterPatterns,
    timeoutReason: _common.HasEsDataFailureReason.remoteDataTimeout
  });
  if (remoteDataResponse) {
    return remoteDataResponse;
  }
  return res.ok({
    body: {
      hasEsData: false
    }
  });
};
exports.createHandler = createHandler;
const timeoutMessage = 'Timeout while checking for Elasticsearch data';
const errorMessage = 'Error while checking for Elasticsearch data';
const hasEsData = async ({
  elasticsearchClient,
  logger,
  res,
  matchPatterns,
  hasEsDataTimeout,
  timeoutReason
}) => {
  try {
    const response = await elasticsearchClient.indices.resolveCluster({
      name: matchPatterns,
      allow_no_indices: true,
      ignore_unavailable: true
    }, {
      requestTimeout: hasEsDataTimeout === 0 ? undefined : hasEsDataTimeout
    });
    const hasData = Object.values(response).some(cluster => cluster.matching_indices);
    if (hasData) {
      return res.ok({
        body: {
          hasEsData: true
        }
      });
    }
  } catch (e) {
    if (e.name === 'TimeoutError') {
      const warningMessage = `${timeoutMessage}: ${timeoutReason}. Current timeout value is ${hasEsDataTimeout}ms. ` + `Use "data_views.hasEsDataTimeout" in kibana.yml to change it, or set to 0 to disable timeouts.`;
      logger.warn(warningMessage);
      return res.customError({
        statusCode: 504,
        body: {
          message: timeoutMessage,
          attributes: {
            failureReason: timeoutReason
          }
        }
      });
    }
    logger.error(e);
    return res.customError({
      statusCode: 500,
      body: {
        message: errorMessage,
        attributes: {
          failureReason: _common.HasEsDataFailureReason.unknown
        }
      }
    });
  }
};
const registerHasEsDataRoute = (router, logger, hasEsDataTimeout) => {
  router.versioned.get({
    path: '/internal/data_views/has_es_data',
    access: 'internal',
    security: {
      authz: {
        enabled: false,
        reason: 'Authorization provided by Elasticsearch'
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      response: {
        200: {
          body: () => _configSchema.schema.object({
            hasEsData: _configSchema.schema.boolean()
          })
        },
        400: {
          body: () => _configSchema.schema.object({
            message: _configSchema.schema.string(),
            attributes: _configSchema.schema.object({
              failureReason: _configSchema.schema.string()
            })
          })
        }
      }
    }
  }, createHandler(logger, hasEsDataTimeout));
};
exports.registerHasEsDataRoute = registerHasEsDataRoute;