"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addFile = void 0;
var _server = require("@kbn/core/server");
var _domain = require("../../../common/types/domain");
var _models = require("../../common/models");
var _error = require("../../common/error");
var _validators = require("../../common/validators");
var _files = require("../../../common/files");
var _authorization = require("../../authorization");
var _validators2 = require("./validators");
var _utils = require("../utils");
var _runtime_types = require("../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Create a file attachment to a case.
 */
const addFile = async (addFileArgs, clientArgs, casesClient) => {
  const {
    caseId,
    file,
    filename,
    mimeType,
    $abort
  } = addFileArgs;
  const {
    logger,
    authorization,
    persistableStateAttachmentTypeRegistry,
    externalReferenceAttachmentTypeRegistry,
    services: {
      userActionService
    },
    fileService
  } = clientArgs;
  let createdFile;
  try {
    (0, _runtime_types.decodeWithExcessOrThrow)(_domain.FileAttachmentMetadataPayloadRt)({
      filename,
      mimeType
    });

    // This will perform an authorization check to ensure the user has access to the parent case
    const theCase = await casesClient.cases.get({
      id: caseId,
      includeComments: false
    });
    const owner = theCase.owner;
    await (0, _validators.validateMaxUserActions)({
      caseId,
      userActionService,
      userActionsToAdd: 1
    });
    const savedObjectID = _server.SavedObjectsUtils.generateId();
    await authorization.ensureAuthorized({
      operation: _authorization.Operations.createComment,
      entities: [{
        owner,
        id: savedObjectID
      }]
    });
    createdFile = await fileService.create({
      name: filename,
      mime: mimeType,
      fileKind: (0, _files.constructFileKindIdByOwner)(owner),
      meta: {
        caseIds: [caseId],
        owner: [owner]
      }
    });
    await createdFile.uploadContent(file, $abort);
    const commentReq = (0, _utils.buildAttachmentRequestFromFileJSON)({
      owner,
      fileMetadata: createdFile.toJSON()
    });
    (0, _validators2.validateRegisteredAttachments)({
      query: commentReq,
      persistableStateAttachmentTypeRegistry,
      externalReferenceAttachmentTypeRegistry
    });
    const createdDate = new Date().toISOString();
    const model = await _models.CaseCommentModel.create(caseId, clientArgs);
    const updatedModel = await model.createComment({
      createdDate,
      commentReq,
      id: savedObjectID
    });
    return await updatedModel.encodeWithComments();
  } catch (error) {
    var _createdFile;
    if ((_createdFile = createdFile) !== null && _createdFile !== void 0 && _createdFile.id) {
      await fileService.delete({
        id: createdFile.id
      });
    }
    throw (0, _error.createCaseError)({
      message: `Failed while adding a comment to case id: ${caseId} error: ${error}`,
      error,
      logger
    });
  }
};
exports.addFile = addFile;