"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SettingsEditor = void 0;
var _lodash = require("lodash");
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _eui = require("@elastic/eui");
var _settings_group = require("./settings_group");
var _settings_form_row = require("./settings_form_row");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEBOUNCE_DELAY = 500;
const ON_LABEL = _i18n.i18n.translate('console.settingsPage.onLabel', {
  defaultMessage: 'On'
});
const OFF_LABEL = _i18n.i18n.translate('console.settingsPage.offLabel', {
  defaultMessage: 'Off'
});
const onceTimeInterval = () => _i18n.i18n.translate('console.settingsPage.refreshInterval.onceTimeInterval', {
  defaultMessage: 'Once, when console loads'
});
const everyNMinutesTimeInterval = value => _i18n.i18n.translate('console.settingsPage.refreshInterval.everyNMinutesTimeInterval', {
  defaultMessage: 'Every {value} {value, plural, one {minute} other {minutes}}',
  values: {
    value
  }
});
const everyHourTimeInterval = () => _i18n.i18n.translate('console.settingsPage.refreshInterval.everyHourTimeInterval', {
  defaultMessage: 'Every hour'
});
const PRESETS_IN_MINUTES = [0, 1, 10, 20, 60];
const intervalOptions = PRESETS_IN_MINUTES.map(value => ({
  value: (value * 60000).toString(),
  inputDisplay: value === 0 ? onceTimeInterval() : value === 60 ? everyHourTimeInterval() : everyNMinutesTimeInterval(value)
}));
const SettingsEditor = props => {
  const isMounted = (0, _react.useRef)(false);
  const [fontSize, setFontSize] = (0, _react.useState)(props.settings.fontSize);
  const [wrapMode, setWrapMode] = (0, _react.useState)(props.settings.wrapMode);
  const [fields, setFields] = (0, _react.useState)(props.settings.autocomplete.fields);
  const [indices, setIndices] = (0, _react.useState)(props.settings.autocomplete.indices);
  const [templates, setTemplates] = (0, _react.useState)(props.settings.autocomplete.templates);
  const [dataStreams, setDataStreams] = (0, _react.useState)(props.settings.autocomplete.dataStreams);
  const [polling, setPolling] = (0, _react.useState)(props.settings.polling);
  const [pollInterval, setPollInterval] = (0, _react.useState)(props.settings.pollInterval);
  const [tripleQuotes, setTripleQuotes] = (0, _react.useState)(props.settings.tripleQuotes);
  const [isHistoryEnabled, setIsHistoryEnabled] = (0, _react.useState)(props.settings.isHistoryEnabled);
  const [isKeyboardShortcutsEnabled, setIsKeyboardShortcutsEnabled] = (0, _react.useState)(props.settings.isKeyboardShortcutsEnabled);
  const [isAccessibilityOverlayEnabled, setIsAccessibilityOverlayEnabled] = (0, _react.useState)(props.settings.isAccessibilityOverlayEnabled);
  const [selectedHost, setSelectedHost] = (0, _react.useState)(props.settings.selectedHost);
  const [availableHosts, setAvailableHosts] = (0, _react.useState)([]);

  // Get available hosts from esHostService after it's initialized
  (0, _react.useEffect)(() => {
    const loadHosts = async () => {
      if (props.esHostService) {
        await props.esHostService.waitForInitialization();
        const hosts = props.esHostService.getAllHosts();
        setAvailableHosts(hosts);
        // If no host is selected, default to the first one
        if (!props.settings.selectedHost && hosts.length > 0) {
          setSelectedHost(hosts[0]);
        }
      }
    };
    loadHosts();
  }, [props.esHostService, props.settings.selectedHost]);
  const autoCompleteCheckboxes = [{
    id: 'fields',
    label: _i18n.i18n.translate('console.settingsPage.fieldsLabelText', {
      defaultMessage: 'Fields'
    }),
    stateSetter: setFields,
    checked: fields
  }, {
    id: 'indices',
    label: _i18n.i18n.translate('console.settingsPage.indicesAndAliasesLabelText', {
      defaultMessage: 'Indices and aliases'
    }),
    stateSetter: setIndices,
    checked: indices
  }, {
    id: 'templates',
    label: _i18n.i18n.translate('console.settingsPage.templatesLabelText', {
      defaultMessage: 'Templates'
    }),
    stateSetter: setTemplates,
    checked: templates
  }, {
    id: 'dataStreams',
    label: _i18n.i18n.translate('console.settingsPage.dataStreamsLabelText', {
      defaultMessage: 'Data streams'
    }),
    stateSetter: setDataStreams,
    checked: dataStreams
  }];
  const saveSettings = () => {
    props.onSaveSettings({
      fontSize,
      wrapMode,
      autocomplete: {
        fields,
        indices,
        templates,
        dataStreams
      },
      polling,
      pollInterval,
      tripleQuotes,
      isHistoryEnabled,
      isKeyboardShortcutsEnabled,
      isAccessibilityOverlayEnabled,
      selectedHost
    });
  };
  const debouncedSaveSettings = (0, _lodash.debounce)(saveSettings, DEBOUNCE_DELAY);
  (0, _react.useEffect)(() => {
    if (isMounted.current) {
      debouncedSaveSettings();
    } else {
      isMounted.current = true;
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [fontSize, wrapMode, fields, indices, templates, dataStreams, polling, pollInterval, tripleQuotes, isHistoryEnabled, isKeyboardShortcutsEnabled, isAccessibilityOverlayEnabled, selectedHost]);
  const onPollingIntervalChange = (0, _react.useCallback)(value => {
    const sanitizedValue = parseInt(value, 10);
    setPolling(!!sanitizedValue);
    setPollInterval(sanitizedValue);
  }, []);
  const toggleKeyboardShortcuts = (0, _react.useCallback)(isEnabled => {
    setIsKeyboardShortcutsEnabled(isEnabled);
  }, []);
  const toggleAccessibilityOverlay = (0, _react.useCallback)(isEnabled => setIsAccessibilityOverlayEnabled(isEnabled), []);
  const toggleSavingToHistory = (0, _react.useCallback)(isEnabled => setIsHistoryEnabled(isEnabled), []);
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiTitle, null, /*#__PURE__*/_react.default.createElement("h2", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "console.settingsPage.pageTitle",
    defaultMessage: "Console settings"
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
    color: "subdued"
  }, /*#__PURE__*/_react.default.createElement("p", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "console.settingsPage.pageDescription",
    defaultMessage: "Customize Console to suit your workflow."
  }))), /*#__PURE__*/_react.default.createElement(_settings_group.SettingsGroup, {
    title: _i18n.i18n.translate('console.settingsPage.generalSettingsLabel', {
      defaultMessage: 'General settings'
    })
  }), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.saveRequestsToHistoryLabel', {
      defaultMessage: 'Save requests to history'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    checked: isHistoryEnabled,
    label: isHistoryEnabled ? ON_LABEL : OFF_LABEL,
    onChange: e => toggleSavingToHistory(e.target.checked)
  })), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.enableKeyboardShortcutsLabel', {
      defaultMessage: 'Keyboard shortcuts'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    "data-test-subj": "enableKeyboardShortcuts",
    label: isKeyboardShortcutsEnabled ? ON_LABEL : OFF_LABEL,
    checked: isKeyboardShortcutsEnabled,
    onChange: e => toggleKeyboardShortcuts(e.target.checked)
  })), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.enableAccessibilityOverlayLabel', {
      defaultMessage: 'Accessibility overlay'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    "data-test-subj": "enableA11yOverlay",
    label: isAccessibilityOverlayEnabled ? ON_LABEL : OFF_LABEL,
    checked: isAccessibilityOverlayEnabled,
    onChange: e => toggleAccessibilityOverlay(e.target.checked)
  })), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.elasticsearchHostLabel', {
      defaultMessage: 'Elasticsearch host'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSuperSelect, {
    css: {
      minWidth: '220px'
    },
    compressed: true,
    disabled: availableHosts.length < 2,
    options: availableHosts.map(host => ({
      value: host,
      inputDisplay: host
    })),
    valueOfSelected: selectedHost || (availableHosts.length > 0 ? availableHosts[0] : ''),
    onChange: value => setSelectedHost(value)
  })), /*#__PURE__*/_react.default.createElement(_settings_group.SettingsGroup, {
    title: _i18n.i18n.translate('console.settingsPage.displaySettingsLabel', {
      defaultMessage: 'Display'
    })
  }), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.fontSizeLabel', {
      defaultMessage: 'Font size'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFieldNumber, {
    css: {
      minWidth: '220px'
    },
    compressed: true,
    "data-test-subj": "setting-font-size-input",
    value: fontSize,
    min: 6,
    max: 50,
    onChange: e => {
      const val = parseInt(e.target.value, 10);
      if (!val) return;
      setFontSize(val);
    }
  })), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.wrapLongLinesLabel', {
      defaultMessage: 'Wrap long lines'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    "data-test-subj": "settingsWrapLines",
    label: wrapMode ? ON_LABEL : OFF_LABEL,
    checked: wrapMode,
    onChange: e => setWrapMode(e.target.checked),
    id: "wrapLines"
  })), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.tripleQuotesMessage', {
      defaultMessage: 'Triple quotes in output'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    "data-test-subj": "tripleQuotes",
    label: tripleQuotes ? ON_LABEL : OFF_LABEL,
    checked: tripleQuotes,
    onChange: e => setTripleQuotes(e.target.checked),
    id: "tripleQuotes"
  })), /*#__PURE__*/_react.default.createElement(_settings_group.SettingsGroup, {
    title: _i18n.i18n.translate('console.settingsPage.autocompleteSettingsLabel', {
      defaultMessage: 'Autocomplete'
    })
  }), autoCompleteCheckboxes.map(opts => /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    key: opts.id,
    label: opts.label
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSwitch, {
    "data-test-subj": `autocomplete-settings-${opts.id}`,
    label: opts.checked ? ON_LABEL : OFF_LABEL,
    checked: opts.checked,
    onChange: e => opts.stateSetter(e.target.checked)
  }))), (fields || indices || templates || dataStreams) && /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_settings_group.SettingsGroup, {
    title: _i18n.i18n.translate('console.settingsPage.autocompleteRefreshSettingsLabel', {
      defaultMessage: 'Autocomplete refresh'
    }),
    description: _i18n.i18n.translate('console.settingsPage.autocompleteRefreshSettingsDescription', {
      defaultMessage: 'Console refreshes autocomplete suggestions by querying Elasticsearch. Use less frequent refreshes to reduce bandwidth costs.'
    })
  }), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.refreshingDataLabel', {
      defaultMessage: 'Refresh frequency'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSuperSelect, {
    css: {
      minWidth: '220px'
    },
    compressed: true,
    options: intervalOptions,
    valueOfSelected: pollInterval.toString(),
    onChange: onPollingIntervalChange
  })), /*#__PURE__*/_react.default.createElement(_settings_form_row.SettingsFormRow, {
    label: _i18n.i18n.translate('console.settingsPage.manualRefreshLabel', {
      defaultMessage: 'Manually refresh autocomplete suggestions'
    })
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
    iconType: "refresh",
    size: "s",
    "data-test-subj": "autocompletePolling",
    id: "autocompletePolling",
    onClick: () => {
      // Only refresh the currently selected settings.
      props.refreshAutocompleteSettings({
        fields,
        indices,
        templates,
        dataStreams
      });
    }
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    defaultMessage: "Refresh",
    id: "console.settingsPage.refreshButtonLabel"
  })))));
};
exports.SettingsEditor = SettingsEditor;