"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLatestFindingsGrouping = exports.isFindingsRootGroupingAggregation = void 0;
var _grouping = require("@kbn/grouping");
var _src = require("@kbn/grouping/src");
var _react = require("react");
var _esQuery = require("@kbn/es-query");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _use_get_benchmark_rules_state_api = require("@kbn/cloud-security-posture/src/hooks/use_get_benchmark_rules_state_api");
var _constants = require("../../../common/constants");
var _data_view_context = require("../../../common/contexts/data_view_context");
var _use_grouped_findings = require("./use_grouped_findings");
var _constants2 = require("./constants");
var _cloud_security_grouping = require("../../../components/cloud_security_grouping");
var _get_filters = require("../utils/get_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getTermAggregation = (key, field) => ({
  [key]: {
    terms: {
      field,
      size: 1
    }
  }
});
const getAggregationsByGroupField = field => {
  if ((0, _src.isNoneGroup)([field])) {
    return [];
  }
  const aggMetrics = [{
    groupByField: {
      cardinality: {
        field
      }
    },
    failedFindings: {
      filter: {
        term: {
          'result.evaluation': {
            value: 'failed'
          }
        }
      }
    },
    passedFindings: {
      filter: {
        term: {
          'result.evaluation': {
            value: 'passed'
          }
        }
      }
    },
    complianceScore: {
      bucket_script: {
        buckets_path: {
          passed: 'passedFindings>_count',
          failed: 'failedFindings>_count'
        },
        script: 'params.passed / (params.passed + params.failed)'
      }
    }
  }];
  switch (field) {
    case _constants.FINDINGS_GROUPING_OPTIONS.RESOURCE_ID:
      return [...aggMetrics, getTermAggregation('resourceName', 'resource.name'), getTermAggregation('resourceSubType', 'resource.sub_type')];
    case _constants.FINDINGS_GROUPING_OPTIONS.RULE_NAME:
      return [...aggMetrics, getTermAggregation('benchmarkName', 'rule.benchmark.name'), getTermAggregation('benchmarkVersion', 'rule.benchmark.version')];
    case _constants.FINDINGS_GROUPING_OPTIONS.CLOUD_ACCOUNT_ID:
      return [...aggMetrics, getTermAggregation('benchmarkName', 'rule.benchmark.name'), getTermAggregation('benchmarkId', 'rule.benchmark.id'), getTermAggregation('accountName', 'cloud.account.name')];
    case _constants.FINDINGS_GROUPING_OPTIONS.ORCHESTRATOR_CLUSTER_ID:
      return [...aggMetrics, getTermAggregation('benchmarkName', 'rule.benchmark.name'), getTermAggregation('benchmarkId', 'rule.benchmark.id'), getTermAggregation('clusterName', 'orchestrator.cluster.name')];
  }
  return aggMetrics;
};

/**
 * Get runtime mappings for the given group field
 * Some fields require additional runtime mappings to aggregate additional information
 * Fallback to keyword type to support custom fields grouping
 */
const getRuntimeMappingsByGroupField = field => {
  if (_constants.CDR_MISCONFIGURATION_GROUPING_RUNTIME_MAPPING_FIELDS !== null && _constants.CDR_MISCONFIGURATION_GROUPING_RUNTIME_MAPPING_FIELDS !== void 0 && _constants.CDR_MISCONFIGURATION_GROUPING_RUNTIME_MAPPING_FIELDS[field]) {
    return _constants.CDR_MISCONFIGURATION_GROUPING_RUNTIME_MAPPING_FIELDS[field].reduce((acc, runtimeField) => ({
      ...acc,
      [runtimeField]: {
        type: 'keyword'
      }
    }), {});
  }
  return {};
};

/**
 * Type Guard for checking if the given source is a FindingsRootGroupingAggregation
 */
const isFindingsRootGroupingAggregation = groupData => {
  var _groupData$passedFind, _groupData$failedFind;
  return (groupData === null || groupData === void 0 ? void 0 : (_groupData$passedFind = groupData.passedFindings) === null || _groupData$passedFind === void 0 ? void 0 : _groupData$passedFind.doc_count) !== undefined && (groupData === null || groupData === void 0 ? void 0 : (_groupData$failedFind = groupData.failedFindings) === null || _groupData$failedFind === void 0 ? void 0 : _groupData$failedFind.doc_count) !== undefined;
};

/**
 * Utility hook to get the latest findings grouping data
 * for the findings page
 */
exports.isFindingsRootGroupingAggregation = isFindingsRootGroupingAggregation;
const useLatestFindingsGrouping = ({
  groupPanelRenderer,
  getGroupStats,
  groupingLevel = 0,
  groupFilters = [],
  selectedGroup
}) => {
  var _groupData$passedFind2, _groupData$failedFind2, _groupData$unitsCount;
  const {
    dataView
  } = (0, _data_view_context.useDataViewContext)();
  const {
    activePageIndex,
    grouping,
    pageSize,
    query,
    onChangeGroupsItemsPerPage,
    onChangeGroupsPage,
    urlQuery,
    setUrlQuery,
    uniqueValue,
    isNoneSelected,
    onResetFilters,
    error,
    filters,
    setActivePageIndex
  } = (0, _cloud_security_grouping.useCloudSecurityGrouping)({
    dataView,
    groupingTitle: _constants2.groupingTitle,
    defaultGroupingOptions: _constants2.defaultGroupingOptions,
    getDefaultQuery: _constants2.getDefaultQuery,
    unit: _constants2.FINDINGS_UNIT,
    groupPanelRenderer,
    getGroupStats,
    groupingLocalStorageKey: _constants.LOCAL_STORAGE_FINDINGS_GROUPING_KEY,
    groupingLevel,
    groupsUnit: _constants2.MISCONFIGURATIONS_GROUPS_UNIT
  });
  const additionalFilters = (0, _esQuery.buildEsQuery)(dataView, [], groupFilters);
  const currentSelectedGroup = selectedGroup || grouping.selectedGroups[0];
  const {
    data: rulesStates
  } = (0, _use_get_benchmark_rules_state_api.useGetCspBenchmarkRulesStatesApi)();
  const mutedRulesFilterQuery = rulesStates ? (0, _cloudSecurityPostureCommon.buildMutedRulesFilter)(rulesStates) : [];
  const groupingQuery = (0, _grouping.getGroupingQuery)({
    additionalFilters: query ? [query, additionalFilters] : [additionalFilters],
    groupByField: currentSelectedGroup,
    uniqueValue,
    timeRange: {
      from: `now-${_cloudSecurityPostureCommon.LATEST_FINDINGS_RETENTION_POLICY}`,
      to: 'now'
    },
    pageNumber: activePageIndex * pageSize,
    size: pageSize,
    sort: [{
      groupByField: {
        order: 'desc'
      }
    }, {
      complianceScore: {
        order: 'asc'
      }
    }],
    statsAggregations: getAggregationsByGroupField(currentSelectedGroup),
    runtimeMappings: getRuntimeMappingsByGroupField(currentSelectedGroup),
    rootAggregations: [{
      failedFindings: {
        filter: {
          term: {
            'result.evaluation': {
              value: 'failed'
            }
          }
        }
      },
      passedFindings: {
        filter: {
          term: {
            'result.evaluation': {
              value: 'passed'
            }
          }
        }
      },
      ...(!(0, _src.isNoneGroup)([currentSelectedGroup]) && {
        nullGroupItems: {
          missing: {
            field: currentSelectedGroup
          }
        }
      })
    }]
  });
  const filteredGroupingQuery = {
    ...groupingQuery,
    query: {
      ...groupingQuery.query,
      bool: {
        ...groupingQuery.query.bool,
        must_not: mutedRulesFilterQuery
      }
    }
  };
  const {
    data,
    isFetching
  } = (0, _use_grouped_findings.useGroupedFindings)({
    query: filteredGroupingQuery,
    enabled: !isNoneSelected
  });
  const groupData = (0, _react.useMemo)(() => (0, _src.parseGroupingQuery)(currentSelectedGroup, uniqueValue, data), [data, currentSelectedGroup, uniqueValue]);
  const totalPassedFindings = isFindingsRootGroupingAggregation(groupData) ? (groupData === null || groupData === void 0 ? void 0 : (_groupData$passedFind2 = groupData.passedFindings) === null || _groupData$passedFind2 === void 0 ? void 0 : _groupData$passedFind2.doc_count) || 0 : 0;
  const totalFailedFindings = isFindingsRootGroupingAggregation(groupData) ? (groupData === null || groupData === void 0 ? void 0 : (_groupData$failedFind2 = groupData.failedFindings) === null || _groupData$failedFind2 === void 0 ? void 0 : _groupData$failedFind2.doc_count) || 0 : 0;
  const onDistributionBarClick = evaluation => {
    setUrlQuery({
      filters: (0, _get_filters.getFilters)({
        filters,
        dataView,
        field: 'result.evaluation',
        value: evaluation,
        negate: false
      })
    });
  };
  const isEmptyResults = !isFetching && isFindingsRootGroupingAggregation(groupData) && !((_groupData$unitsCount = groupData.unitsCount) !== null && _groupData$unitsCount !== void 0 && _groupData$unitsCount.value);
  return {
    groupData,
    grouping,
    isFetching,
    activePageIndex,
    setActivePageIndex,
    pageSize,
    selectedGroup,
    onChangeGroupsItemsPerPage,
    onChangeGroupsPage,
    urlQuery,
    setUrlQuery,
    isGroupSelected: !isNoneSelected,
    isGroupLoading: !data,
    onResetFilters,
    filters,
    error,
    onDistributionBarClick,
    totalPassedFindings,
    totalFailedFindings,
    isEmptyResults
  };
};
exports.useLatestFindingsGrouping = useLatestFindingsGrouping;