"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaMcpHttpTransport = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _types = require("@modelcontextprotocol/sdk/types.js");
var _nodeCrypto = require("node:crypto");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Server transport for Streamable HTTP: this implements the MCP Streamable HTTP transport specification.
 * It supports direct HTTP responses. It doesn't support SSE streaming.
 *
 * It is compatible with Kibana's HTTP request/response abstractions. Implementation is adapted from:
 * https://github.com/modelcontextprotocol/typescript-sdk/blob/main/src/server/streamableHttp.ts
 *
 * In stateful mode:
 * - Session ID is generated and included in response headers
 * - Session ID is always included in initialization responses
 * - Requests with invalid session IDs are rejected with 404 Not Found
 * - Non-initialization requests without a session ID are rejected with 400 Bad Request
 * - State is maintained in-memory (connections, message history)
 *
 * In stateless mode:
 * - No Session ID is included in any responses
 * - No session validation is performed
 */
class KibanaMcpHttpTransport {
  constructor(options) {
    // when sessionId is not set (undefined), it means the transport is in stateless mode
    (0, _defineProperty2.default)(this, "sessionIdGenerator", void 0);
    (0, _defineProperty2.default)(this, "_started", false);
    (0, _defineProperty2.default)(this, "_streamMapping", new Map());
    (0, _defineProperty2.default)(this, "_requestToStreamMapping", new Map());
    (0, _defineProperty2.default)(this, "_requestResponseMap", new Map());
    (0, _defineProperty2.default)(this, "_initialized", false);
    (0, _defineProperty2.default)(this, "_logger", void 0);
    (0, _defineProperty2.default)(this, "_responseCallbacks", new Map());
    (0, _defineProperty2.default)(this, "sessionId", void 0);
    (0, _defineProperty2.default)(this, "onclose", void 0);
    (0, _defineProperty2.default)(this, "onerror", void 0);
    (0, _defineProperty2.default)(this, "onmessage", void 0);
    this.sessionIdGenerator = options.sessionIdGenerator;
    this._logger = options.logger;
  }

  /**
   * Registers a callback for when a response is ready for a specific stream
   */
  registerResponseCallback(streamId, callback) {
    this._responseCallbacks.set(streamId, callback);
  }

  /**
   * Handles an incoming HTTP request, whether GET or POST
   */
  async handleRequest(req, res) {
    if (req.route.method === 'post') {
      return await this.handlePostRequest(req, res);
    } else {
      return await this.handleUnsupportedRequest(res);
    }
  }

  /**
   * Starts the transport. This is required by the Transport interface but is a no-op
   * for the Streamable HTTP transport as connections are managed per-request.
   */
  async start() {
    if (this._started) {
      throw new Error('Transport already started');
    }
    this._started = true;
  }

  /**
   * Closes the transport and cleans up resources
   */
  async close() {
    var _this$onclose;
    this._streamMapping.clear();

    // Clear any pending responses
    this._requestResponseMap.clear();
    this._requestToStreamMapping.clear();
    this._responseCallbacks.clear();

    // Clear session state
    this.sessionId = undefined;
    this._initialized = false;
    (_this$onclose = this.onclose) === null || _this$onclose === void 0 ? void 0 : _this$onclose.call(this);
  }

  /**
   * Handles POST requests containing JSON-RPC messages compatible with MCP specification
   */
  async handlePostRequest(request, responseFactory) {
    try {
      this._logger.debug('Processing POST request');

      // Validate the Accept header
      const acceptHeader = request.headers.accept;

      // The client MUST include an Accept header, listing application/json as supported content type.
      if (!(acceptHeader !== null && acceptHeader !== void 0 && acceptHeader.includes('application/json'))) {
        this._logger.warn('Request rejected: Invalid Accept header');
        return responseFactory.customError({
          statusCode: 406,
          body: JSON.stringify({
            jsonrpc: '2.0',
            error: {
              code: _types.ErrorCode.ConnectionClosed,
              message: 'Not Acceptable: Client must accept application/json'
            },
            id: null
          })
        });
      }
      const ct = request.headers['content-type'];
      if (!ct || !ct.includes('application/json')) {
        this._logger.warn('Request rejected: Invalid Content-Type');
        return responseFactory.customError({
          statusCode: 415,
          body: JSON.stringify({
            jsonrpc: '2.0',
            error: {
              code: _types.ErrorCode.ConnectionClosed,
              message: 'Unsupported Media Type: Content-Type must be application/json'
            },
            id: null
          })
        });
      }
      const rawMessage = request.body;
      let messages;

      // handle batch and single messages
      if (Array.isArray(rawMessage)) {
        this._logger.debug(`Processing batch request with ${rawMessage.length} messages`);
        messages = rawMessage.map(msg => _types.JSONRPCMessageSchema.parse(msg));
      } else {
        this._logger.debug('Processing single message request');
        messages = [_types.JSONRPCMessageSchema.parse(rawMessage)];
      }

      // Check if this is an initialization request
      const isInitializationRequest = messages.some(_types.isInitializeRequest);
      if (isInitializationRequest) {
        var _this$sessionIdGenera;
        if (this._initialized && this.sessionId !== undefined) {
          this._logger.warn('Initialization request rejected - server already initialized');
          return responseFactory.badRequest({
            body: JSON.stringify({
              jsonrpc: '2.0',
              error: {
                code: _types.ErrorCode.InvalidRequest,
                message: 'Invalid Request: Server already initialized'
              },
              id: null
            })
          });
        }
        if (messages.length > 1) {
          this._logger.warn('Initialization request rejected - multiple messages not allowed');
          return responseFactory.badRequest({
            body: JSON.stringify({
              jsonrpc: '2.0',
              error: {
                code: _types.ErrorCode.InvalidRequest,
                message: 'Invalid Request: Only one initialization request is allowed'
              },
              id: null
            })
          });
        }
        this.sessionId = (_this$sessionIdGenera = this.sessionIdGenerator) === null || _this$sessionIdGenera === void 0 ? void 0 : _this$sessionIdGenera.call(this);
        this._initialized = true;
        this._logger.debug(`Session initialized: ${this.sessionId}`);
      }

      // check if it contains requests
      const hasRequests = messages.some(_types.isJSONRPCRequest);
      if (!hasRequests) {
        this._logger.debug('Processing notifications/responses');
        // if it only contains notifications or responses, return 202
        const response = responseFactory.accepted();

        // handle each message
        for (const message of messages) {
          var _this$onmessage;
          (_this$onmessage = this.onmessage) === null || _this$onmessage === void 0 ? void 0 : _this$onmessage.call(this, message);
        }
        return response;
      } else {
        const streamId = (0, _nodeCrypto.randomUUID)();
        this._logger.debug(`Processing requests with stream ID: ${streamId}`);

        // Create a promise that will resolve when the response is ready
        const responsePromise = new Promise(resolve => {
          this.registerResponseCallback(streamId, resolve);
        });

        // Store the response factory for this request to send messages back through this connection
        for (const message of messages) {
          if ((0, _types.isJSONRPCRequest)(message)) {
            this._logger.debug(`Mapping request ${message.id} to stream ${streamId}`);
            this._streamMapping.set(streamId, responseFactory);
            this._requestToStreamMapping.set(message.id, streamId);
          }
        }

        // Set up close handler for client disconnects
        request.events.aborted$.subscribe(() => {
          this._logger.debug(`Stream ${streamId} aborted by client`);
          this._streamMapping.delete(streamId);
          this._responseCallbacks.delete(streamId);
        });

        // handle each message
        for (const message of messages) {
          var _this$onmessage2;
          (_this$onmessage2 = this.onmessage) === null || _this$onmessage2 === void 0 ? void 0 : _this$onmessage2.call(this, message);
        }

        // Wait for the response to be ready
        return await responsePromise;
      }
    } catch (error) {
      var _this$onerror;
      this._logger.error(`Request processing error: ${error}`);
      (_this$onerror = this.onerror) === null || _this$onerror === void 0 ? void 0 : _this$onerror.call(this, error);
      return responseFactory.badRequest({
        body: JSON.stringify({
          jsonrpc: '2.0',
          error: {
            code: _types.ErrorCode.ParseError,
            message: 'Parse error',
            data: String(error)
          },
          id: null
        })
      });
    }
  }

  /**
   * Sends a JSON-RPC message
   */
  async send(message, options) {
    try {
      this._logger.debug('Processing outgoing message');
      let requestId = options === null || options === void 0 ? void 0 : options.relatedRequestId;
      if ((0, _types.isJSONRPCResponse)(message) || (0, _types.isJSONRPCError)(message)) {
        requestId = message.id;
      }
      if (requestId === undefined) {
        this._logger.debug('Processing notification message');
        // todo: handle notifications
        return;
      }

      // Get the response for this request
      const streamId = this._requestToStreamMapping.get(requestId);
      const responseFactory = this._streamMapping.get(streamId);
      if (!streamId) {
        const error = `No connection established for request ID: ${String(requestId)}`;
        this._logger.error(error);
        throw new Error(error);
      }
      if ((0, _types.isJSONRPCResponse)(message) || (0, _types.isJSONRPCError)(message)) {
        this._logger.debug(`Processing response for request ${String(requestId)}`);
        this._requestResponseMap.set(requestId, message);
        const relatedIds = Array.from(this._requestToStreamMapping.entries()).filter(([_, sid]) => this._streamMapping.get(sid) === responseFactory).map(([id]) => id);

        // Check if we have responses for all requests using this connection
        const allResponsesReady = relatedIds.every(id => this._requestResponseMap.has(id));
        if (allResponsesReady) {
          if (!responseFactory) {
            const error = `No connection established for request ID: ${String(requestId)}`;
            this._logger.error(error);
            throw new Error(error);
          }

          // All responses ready, send as JSON
          const headers = {
            'Content-Type': 'application/json'
          };
          if (this.sessionId !== undefined) {
            headers['mcp-session-id'] = this.sessionId;
          }
          const responses = relatedIds.map(id => this._requestResponseMap.get(id));
          const responseBody = responses.length === 1 ? JSON.stringify(responses[0]) : JSON.stringify(responses);
          const response = responseFactory.ok({
            headers,
            body: responseBody
          });

          // Trigger the response callback
          const callback = this._responseCallbacks.get(streamId);
          if (callback) {
            callback(response);
            this._responseCallbacks.delete(streamId);
          }

          // Clean up
          for (const id of relatedIds) {
            this._requestResponseMap.delete(id);
            this._requestToStreamMapping.delete(id);
          }
          this._streamMapping.delete(streamId);
          this._logger.debug('Request mappings cleaned up');
        }
      }
    } catch (error) {
      var _this$onerror2;
      this._logger.error(`Error sending message: ${error}`);
      (_this$onerror2 = this.onerror) === null || _this$onerror2 === void 0 ? void 0 : _this$onerror2.call(this, error);
    }
  }

  /**
   * Handles unsupported requests (GET, DELETE, PUT, PATCH, etc.)
   */
  async handleUnsupportedRequest(res) {
    return res.customError({
      statusCode: 405,
      body: JSON.stringify({
        jsonrpc: '2.0',
        error: {
          code: _types.ErrorCode.InvalidRequest,
          message: 'Method not allowed.'
        },
        id: null
      })
    });
  }
}
exports.KibanaMcpHttpTransport = KibanaMcpHttpTransport;