"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeUrlState = exports.hydrateDataSourceSelection = exports.getDiscoverFiltersFromState = exports.getDiscoverColumnsWithFallbackFieldsFromDisplayOptions = void 0;
var _discoverUtils = require("@kbn/discover-utils");
var _esQuery = require("@kbn/es-query");
var _src = require("@kbn/discover-utils/src");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const hydrateDataSourceSelection = dataSourceSelection => {
  if (dataSourceSelection.selectionType === 'all') {
    return {
      id: _src.ALL_LOGS_DATA_VIEW_ID
    };
  } else if (dataSourceSelection.selectionType === 'single' && dataSourceSelection.selection) {
    const selection = dataSourceSelection.selection;
    return {
      id: `dataset-${selection.dataset.name}`,
      name: selection.title || selection.dataset.name,
      title: selection.dataset.indexPattern || selection.dataset.name,
      timeFieldName: '@timestamp'
    };
  } else if (dataSourceSelection.selectionType === 'dataView' && dataSourceSelection.selection) {
    const selection = dataSourceSelection.selection;
    return {
      id: selection.dataView.id,
      name: selection.dataView.title,
      title: selection.dataView.indexPattern || selection.dataView.title,
      timeFieldName: '@timestamp'
    };
  } else if (dataSourceSelection.selectionType === 'unresolved' && dataSourceSelection.selection) {
    const selection = dataSourceSelection.selection;
    return {
      id: `dataset-${selection.dataset.name}`,
      name: selection.name || selection.dataset.name,
      title: selection.dataset.indexPattern || selection.dataset.name,
      timeFieldName: '@timestamp'
    };
  }
  return {
    id: 'discover-observability-solution-all-logs',
    name: 'All logs',
    title: 'logs-*,dataset-logs-*-*',
    timeFieldName: '@timestamp'
  };
};
exports.hydrateDataSourceSelection = hydrateDataSourceSelection;
const normalizeUrlState = input => {
  try {
    const parsed = parseUnknownInput(input);
    const normalized = normalizeVersions(parsed);
    return convertToPublicState(normalized);
  } catch (err) {
    return null;
  }
};
exports.normalizeUrlState = normalizeUrlState;
const parseUnknownInput = input => {
  if (typeof input !== 'object' || input === null) {
    throw new Error('Invalid URL state format');
  }
  return input;
};
const normalizeVersions = schema => {
  if ('datasetSelection' in schema) {
    return {
      ...schema,
      v: 2,
      dataSourceSelection: schema.datasetSelection
    };
  }
  return schema;
};
const convertToPublicState = schema => ({
  chart: schema.breakdownField ? {
    breakdownField: schema.breakdownField
  } : undefined,
  controls: schema.controls,
  dataSourceSelection: schema.dataSourceSelection,
  filters: schema.filters,
  grid: {
    columns: schema.columns,
    rows: {
      rowHeight: schema.rowHeight,
      rowsPerPage: schema.rowsPerPage
    }
  },
  query: schema.query,
  refreshInterval: schema.refreshInterval,
  time: schema.time
});
const getDiscoverColumnsWithFallbackFieldsFromDisplayOptions = displayOptions => {
  var _displayOptions$grid$;
  return displayOptions === null || displayOptions === void 0 ? void 0 : (_displayOptions$grid$ = displayOptions.grid.columns) === null || _displayOptions$grid$ === void 0 ? void 0 : _displayOptions$grid$.flatMap(column => {
    if (column.type === 'document-field' && column.field) {
      return column.field;
    } else if (column.type === 'smart-field' && column.smartField) {
      return _discoverUtils.SMART_FALLBACK_FIELDS[column.smartField].fallbackFields;
    }
    return [];
  });
};
exports.getDiscoverColumnsWithFallbackFieldsFromDisplayOptions = getDiscoverColumnsWithFallbackFieldsFromDisplayOptions;
const createDiscoverPhrasesFilter = ({
  key,
  values,
  negate,
  index
}) => ({
  meta: {
    index,
    type: _esQuery.FILTERS.PHRASES,
    key,
    params: values.map(value => value.toString()),
    negate
  },
  query: {
    bool: {
      should: values.map(value => ({
        match_phrase: {
          [key]: value.toString()
        }
      })),
      minimum_should_match: 1
    }
  }
});
const createDiscoverExistsFilter = ({
  index,
  key,
  negate
}) => ({
  meta: {
    index,
    type: _esQuery.FILTERS.EXISTS,
    value: _esQuery.FILTERS.EXISTS,
    key,
    negate
  },
  query: {
    exists: {
      field: key
    }
  }
});
const getDiscoverFiltersFromState = (index, filters = [], controls) => {
  return [...filters, ...(controls ? Object.entries(controls).reduce((acc, [key, control]) => {
    if (control.selection.type === 'exists') {
      acc.push(createDiscoverExistsFilter({
        index,
        key,
        negate: control.mode === 'exclude'
      }));
    } else if (control.selection.selectedOptions.length > 0) {
      acc.push(createDiscoverPhrasesFilter({
        index,
        key,
        values: control.selection.selectedOptions,
        negate: control.mode === 'exclude'
      }));
    }
    return acc;
  }, []) : [])];
};
exports.getDiscoverFiltersFromState = getDiscoverFiltersFromState;