"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRandomSamplerWrapper = void 0;
exports.isValidProbability = isValidProbability;
var _lodash = require("lodash");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _get_sample_probability = require("./get_sample_probability");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_AGG_NAME = 'sample';
/**
 * Type guard for RandomSamplerOptionProbability
 */
function isRandomSamplerOptionProbability(arg) {
  return (0, _mlIsPopulatedObject.isPopulatedObject)(arg, ['probability']);
}
/**
 * Check if a given probability is suitable for the `random_sampler` aggregation.
 * @param {unknown} p The probability to be tested.
 * @returns {boolean}
 */
function isValidProbability(p) {
  return typeof p === 'number' && p > 0 && p <= 0.5;
}

/**
 * The return type of the `createRandomSamplerWrapper` factory.
 */

/**
 * Factory to create the random sampler wrapper utility.
 * @param {RandomSamplerOptions} options RandomSamplerOptions
 * @returns {RandomSamplerWrapper} random sampler wrapper utility
 */
const createRandomSamplerWrapper = options => {
  var _options$aggName;
  const probability = isRandomSamplerOptionProbability(options) ? options.probability : (0, _get_sample_probability.getSampleProbability)(options.totalNumDocs);
  const aggName = (_options$aggName = options.aggName) !== null && _options$aggName !== void 0 ? _options$aggName : DEFAULT_AGG_NAME;
  const wrap = aggs => {
    if (!isValidProbability(probability)) {
      return aggs;
    }
    return {
      [aggName]: {
        random_sampler: {
          probability,
          ...(options.seed ? {
            seed: options.seed
          } : {})
        },
        aggs
      }
    };
  };
  const unwrap = responseAggs => !isValidProbability(probability) ? responseAggs : (0, _lodash.get)(responseAggs, [aggName]);
  return {
    wrap,
    unwrap,
    probability
  };
};
exports.createRandomSamplerWrapper = createRandomSamplerWrapper;