"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deserializeLegacyTemplate = deserializeLegacyTemplate;
exports.deserializeLegacyTemplateList = deserializeLegacyTemplateList;
exports.deserializeTemplate = deserializeTemplate;
exports.deserializeTemplateList = deserializeTemplateList;
exports.serializeLegacyTemplate = serializeLegacyTemplate;
exports.serializeTemplate = serializeTemplate;
var _data_stream_utils = require("./data_stream_utils");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const hasEntries = (data = {}) => Object.entries(data).length > 0;
function serializeTemplate(templateDeserialized, dataStreamOptions) {
  var _template$settings;
  const {
    version,
    priority,
    indexPatterns,
    template,
    composedOf,
    ignoreMissingComponentTemplates,
    dataStream,
    indexMode,
    _meta,
    allowAutoCreate,
    deprecated
  } = templateDeserialized;
  return {
    version,
    priority,
    template: {
      ...template,
      settings: {
        ...(template === null || template === void 0 ? void 0 : template.settings),
        index: {
          ...(template === null || template === void 0 ? void 0 : (_template$settings = template.settings) === null || _template$settings === void 0 ? void 0 : _template$settings.index),
          mode: indexMode
        }
      },
      // If the existing template contains data stream options, we need to persist them.
      // Otherwise, they will be lost when the template is updated.
      ...(dataStreamOptions && {
        data_stream_options: dataStreamOptions
      })
    },
    index_patterns: indexPatterns,
    data_stream: dataStream,
    composed_of: composedOf,
    ignore_missing_component_templates: ignoreMissingComponentTemplates,
    allow_auto_create: _constants.allowAutoCreateRadioValues === null || _constants.allowAutoCreateRadioValues === void 0 ? void 0 : _constants.allowAutoCreateRadioValues[allowAutoCreate],
    _meta,
    deprecated
  };
}
function deserializeTemplate(templateEs, cloudManagedTemplatePrefix, isLogsdbEnabled) {
  var _settings$index, _settings$index$lifec, _settings$index$mode, _settings$index2;
  const {
    name,
    version,
    index_patterns: indexPatterns,
    template = {},
    priority,
    _meta,
    composed_of: composedOf,
    ignore_missing_component_templates: ignoreMissingComponentTemplates,
    data_stream: dataStream,
    deprecated,
    allow_auto_create: allowAutoCreate
  } = templateEs;
  const {
    settings
  } = template;
  let type = 'default';
  if (Boolean(cloudManagedTemplatePrefix && name.startsWith(cloudManagedTemplatePrefix))) {
    type = 'cloudManaged';
  } else if (name.startsWith('.')) {
    type = 'system';
  } else if (Boolean((_meta === null || _meta === void 0 ? void 0 : _meta.managed) === true)) {
    type = 'managed';
  }
  const ilmPolicyName = settings === null || settings === void 0 ? void 0 : (_settings$index = settings.index) === null || _settings$index === void 0 ? void 0 : (_settings$index$lifec = _settings$index.lifecycle) === null || _settings$index$lifec === void 0 ? void 0 : _settings$index$lifec.name;
  const indexMode = (_settings$index$mode = settings === null || settings === void 0 ? void 0 : (_settings$index2 = settings.index) === null || _settings$index2 === void 0 ? void 0 : _settings$index2.mode) !== null && _settings$index$mode !== void 0 ? _settings$index$mode : isLogsdbEnabled && indexPatterns.some(pattern => pattern === 'logs-*-*') ? _constants.LOGSDB_INDEX_MODE : _constants.STANDARD_INDEX_MODE;
  const deserializedTemplate = {
    name,
    version,
    priority,
    ...(template.lifecycle ? {
      lifecycle: (0, _data_stream_utils.deserializeESLifecycle)(template.lifecycle)
    } : {}),
    indexPatterns: indexPatterns.sort(),
    template,
    indexMode,
    ilmPolicy: ilmPolicyName ? {
      name: ilmPolicyName
    } : undefined,
    composedOf: composedOf !== null && composedOf !== void 0 ? composedOf : [],
    ignoreMissingComponentTemplates: ignoreMissingComponentTemplates !== null && ignoreMissingComponentTemplates !== void 0 ? ignoreMissingComponentTemplates : [],
    dataStream,
    allowAutoCreate: allowAutoCreate === true ? _constants.allowAutoCreateRadioIds.TRUE_RADIO_OPTION : allowAutoCreate === false ? _constants.allowAutoCreateRadioIds.FALSE_RADIO_OPTION : _constants.allowAutoCreateRadioIds.NO_OVERWRITE_RADIO_OPTION,
    _meta,
    deprecated,
    _kbnMeta: {
      type,
      hasDatastream: Boolean(dataStream)
    }
  };
  return deserializedTemplate;
}
function deserializeTemplateList(indexTemplates, cloudManagedTemplatePrefix) {
  return indexTemplates.map(({
    name,
    index_template: templateSerialized
  }) => {
    const {
      template: {
        mappings,
        settings,
        aliases
      } = {},
      ...deserializedTemplate
    } = deserializeTemplate({
      name,
      ...templateSerialized
    }, cloudManagedTemplatePrefix);
    return {
      ...deserializedTemplate,
      hasSettings: hasEntries(settings),
      hasAliases: hasEntries(aliases),
      hasMappings: hasEntries(mappings)
    };
  });
}

/**
 * ------------------------------------------
 * --------- LEGACY INDEX TEMPLATES ---------
 * ------------------------------------------
 */

function serializeLegacyTemplate(template) {
  const {
    version,
    order,
    indexPatterns,
    template: {
      settings,
      aliases,
      mappings
    } = {}
  } = template;
  return {
    version,
    order,
    index_patterns: indexPatterns,
    settings,
    aliases,
    mappings
  };
}
function deserializeLegacyTemplate(templateEs, cloudManagedTemplatePrefix, isLogsdbEnabled) {
  const {
    settings,
    aliases,
    mappings,
    ...rest
  } = templateEs;
  const deserializedTemplate = deserializeTemplate({
    ...rest,
    template: {
      aliases,
      settings,
      mappings
    }
  }, cloudManagedTemplatePrefix, isLogsdbEnabled);
  return {
    ...deserializedTemplate,
    order: templateEs.order,
    _kbnMeta: {
      ...deserializedTemplate._kbnMeta,
      isLegacy: true
    }
  };
}
function deserializeLegacyTemplateList(indexTemplatesByName, cloudManagedTemplatePrefix) {
  return Object.entries(indexTemplatesByName).map(([name, templateSerialized]) => {
    const {
      template: {
        mappings,
        settings,
        aliases
      } = {},
      ...deserializedTemplate
    } = deserializeLegacyTemplate({
      name,
      ...templateSerialized
    }, cloudManagedTemplatePrefix);
    return {
      ...deserializedTemplate,
      hasSettings: hasEntries(settings),
      hasAliases: hasEntries(aliases),
      hasMappings: hasEntries(mappings)
    };
  });
}