"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRemoteSyncedIntegrationsStatus = exports.getFollowerIndexInfo = exports.fetchAndCompareSyncedIntegrations = void 0;
var _lodash = require("lodash");
var _services = require("../../services");
var _get = require("../../services/epm/packages/get");
var _errors = require("../../errors");
var _types = require("../../../common/types");
var _fleet_synced_integrations = require("../../services/setup/fleet_synced_integrations");
var _custom_assets = require("./custom_assets");
var _sync_integrations_on_remote = require("./sync_integrations_on_remote");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFollowerIndexInfo = async (esClient, logger) => {
  try {
    var _res$follower_indices, _resStats$indices$, _resStats$indices$$sh;
    const index = await (0, _sync_integrations_on_remote.getFollowerIndex)(esClient, new AbortController());
    if (!index) {
      return {
        error: `Follower index not found`
      };
    }
    const res = await esClient.ccr.followInfo({
      index
    });
    if (!(res !== null && res !== void 0 && res.follower_indices) || res.follower_indices.length === 0) return {
      error: `Follower index ${index} not available`
    };
    if (((_res$follower_indices = res.follower_indices[0]) === null || _res$follower_indices === void 0 ? void 0 : _res$follower_indices.status) === 'paused') {
      return {
        error: `Follower index ${index} paused`
      };
    }
    const resStats = await esClient.ccr.followStats({
      index
    });
    if (resStats !== null && resStats !== void 0 && (_resStats$indices$ = resStats.indices[0]) !== null && _resStats$indices$ !== void 0 && (_resStats$indices$$sh = _resStats$indices$.shards[0]) !== null && _resStats$indices$$sh !== void 0 && _resStats$indices$$sh.fatal_exception) {
      var _resStats$indices$0$s;
      return {
        error: `Follower index ${index} fatal exception: ${(_resStats$indices$0$s = resStats.indices[0].shards[0].fatal_exception) === null || _resStats$indices$0$s === void 0 ? void 0 : _resStats$indices$0$s.reason}`
      };
    }
    return {
      info: res.follower_indices[0]
    };
  } catch (err) {
    var _err$body, _err$body$error;
    if ((err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'index_not_found_exception') {
      throw new _errors.IndexNotFoundError(`Index not found`);
    }
    logger.error('error', err === null || err === void 0 ? void 0 : err.message);
    throw err;
  }
};
exports.getFollowerIndexInfo = getFollowerIndexInfo;
const fetchAndCompareSyncedIntegrations = async (esClient, savedObjectsClient, index, logger) => {
  try {
    var _searchRes$hits$hits$, _searchRes$hits$hits$2;
    // find integrations on ccr index
    const searchRes = await esClient.search({
      index,
      sort: [{
        'integrations.updated_at': {
          order: 'desc'
        }
      }]
    });
    if (((_searchRes$hits$hits$ = searchRes.hits.hits[0]) === null || _searchRes$hits$hits$ === void 0 ? void 0 : _searchRes$hits$hits$._source) === undefined) {
      return {
        integrations: [],
        error: `No integrations found on ${index}`
      };
    }
    const ccrIndex = (_searchRes$hits$hits$2 = searchRes.hits.hits[0]) === null || _searchRes$hits$hits$2 === void 0 ? void 0 : _searchRes$hits$hits$2._source;
    const {
      integrations: ccrIntegrations,
      custom_assets: ccrCustomAssets,
      remote_es_hosts: remoteEsHosts
    } = ccrIndex;

    // find integrations installed on remote
    const installedIntegrations = await (0, _get.getPackageSavedObjects)(savedObjectsClient);
    const installedIntegrationsByName = ((installedIntegrations === null || installedIntegrations === void 0 ? void 0 : installedIntegrations.saved_objects) || []).reduce((acc, integration) => {
      if (integration !== null && integration !== void 0 && integration.id) {
        acc[integration.id] = integration;
      }
      return acc;
    }, {});
    const customAssetsStatus = await fetchAndCompareCustomAssets(esClient, logger, ccrCustomAssets, installedIntegrationsByName);
    const isSyncUninstalledEnabled = remoteEsHosts === null || remoteEsHosts === void 0 ? void 0 : remoteEsHosts.some(host => host.sync_uninstalled_integrations);
    const integrationsStatus = compareIntegrations(ccrIntegrations, installedIntegrationsByName, isSyncUninstalledEnabled);
    const result = {
      ...integrationsStatus,
      ...(customAssetsStatus && {
        custom_assets: customAssetsStatus
      })
    };
    return result;
  } catch (error) {
    logger.error('error', error === null || error === void 0 ? void 0 : error.message);
    return {
      integrations: [],
      error: error === null || error === void 0 ? void 0 : error.message
    };
  }
};
exports.fetchAndCompareSyncedIntegrations = fetchAndCompareSyncedIntegrations;
const compareIntegrations = (ccrIntegrations, installedIntegrationsByName, isSyncUninstalledEnabled) => {
  const integrationsStatus = ccrIntegrations === null || ccrIntegrations === void 0 ? void 0 : ccrIntegrations.map(ccrIntegration => {
    const baseIntegrationData = {
      package_name: ccrIntegration.package_name,
      package_version: ccrIntegration.package_version,
      install_status: ccrIntegration.install_status
    };
    const localIntegrationSO = installedIntegrationsByName[ccrIntegration.package_name];
    // Handle case of integration uninstalled from both clusters
    if (isSyncUninstalledEnabled && !(localIntegrationSO !== null && localIntegrationSO !== void 0 && localIntegrationSO.attributes) && ccrIntegration.install_status === 'not_installed') {
      return {
        ...baseIntegrationData,
        install_status: {
          main: 'not_installed',
          remote: 'not_installed'
        },
        sync_status: _types.SyncStatus.COMPLETED,
        updated_at: ccrIntegration === null || ccrIntegration === void 0 ? void 0 : ccrIntegration.updated_at
      };
    }
    if (!localIntegrationSO) {
      return {
        ...baseIntegrationData,
        updated_at: ccrIntegration.updated_at,
        sync_status: _types.SyncStatus.SYNCHRONIZING,
        install_status: {
          main: ccrIntegration.install_status
        }
      };
    }
    if (ccrIntegration.install_status !== 'not_installed' && ccrIntegration.package_version !== (localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.version)) {
      return {
        ...baseIntegrationData,
        updated_at: ccrIntegration.updated_at,
        sync_status: _types.SyncStatus.FAILED,
        install_status: {
          main: ccrIntegration.install_status,
          remote: localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status
        },
        error: `Found incorrect installed version ${localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.version}`
      };
    }
    if (ccrIntegration.install_status !== 'not_installed' && (localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status) === 'install_failed') {
      var _localIntegrationSO$a, _localIntegrationSO$a2;
      let latestFailedAttemptTime = '';
      let latestFailedAttempt = '';
      if (localIntegrationSO !== null && localIntegrationSO !== void 0 && (_localIntegrationSO$a = localIntegrationSO.attributes) !== null && _localIntegrationSO$a !== void 0 && (_localIntegrationSO$a2 = _localIntegrationSO$a.latest_install_failed_attempts) !== null && _localIntegrationSO$a2 !== void 0 && _localIntegrationSO$a2[0]) {
        var _latestInstallFailedA;
        const latestInstallFailedAttempts = localIntegrationSO.attributes.latest_install_failed_attempts[0];
        latestFailedAttemptTime = `at ${new Date(latestInstallFailedAttempts.created_at).toUTCString()}`;

        // handling special case for those integrations that cannot be found in registry
        if (((_latestInstallFailedA = latestInstallFailedAttempts.error) === null || _latestInstallFailedA === void 0 ? void 0 : _latestInstallFailedA.name) === 'PackageNotFoundError') {
          return {
            ...baseIntegrationData,
            install_status: {
              main: ccrIntegration.install_status,
              remote: 'not_installed'
            },
            updated_at: ccrIntegration.updated_at,
            sync_status: _types.SyncStatus.WARNING,
            warning: {
              title: `Integration can't be automatically synced`,
              message: `This integration must be manually installed on the remote cluster. Automatic updates and remote installs are not supported.`
            }
          };
        } else {
          var _latestInstallFailedA2, _latestInstallFailedA3;
          latestFailedAttempt = (_latestInstallFailedA2 = (_latestInstallFailedA3 = latestInstallFailedAttempts.error) === null || _latestInstallFailedA3 === void 0 ? void 0 : _latestInstallFailedA3.message) !== null && _latestInstallFailedA2 !== void 0 ? _latestInstallFailedA2 : '';
        }
      }
      return {
        ...baseIntegrationData,
        install_status: {
          main: ccrIntegration.install_status,
          remote: localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status
        },
        updated_at: ccrIntegration.updated_at,
        sync_status: _types.SyncStatus.FAILED,
        error: `Installation status: ${localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status} ${latestFailedAttempt} ${latestFailedAttemptTime}`
      };
    }
    if (isSyncUninstalledEnabled && ccrIntegration.install_status === 'not_installed' && (localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status) === 'installed') {
      var _localIntegrationSO$a3, _localIntegrationSO$a4;
      let latestUninstallFailedAttemptTime = '';
      let latestUninstallFailedAttempt = '';
      if (localIntegrationSO !== null && localIntegrationSO !== void 0 && (_localIntegrationSO$a3 = localIntegrationSO.attributes) !== null && _localIntegrationSO$a3 !== void 0 && (_localIntegrationSO$a4 = _localIntegrationSO$a3.latest_uninstall_failed_attempts) !== null && _localIntegrationSO$a4 !== void 0 && _localIntegrationSO$a4[0]) {
        var _latestInstallFailedA4, _latestInstallFailedA5;
        const latestInstallFailedAttempts = localIntegrationSO.attributes.latest_uninstall_failed_attempts[0];
        latestUninstallFailedAttemptTime = `at ${new Date(latestInstallFailedAttempts.created_at).toUTCString()}`;
        latestUninstallFailedAttempt = (_latestInstallFailedA4 = (_latestInstallFailedA5 = latestInstallFailedAttempts.error) === null || _latestInstallFailedA5 === void 0 ? void 0 : _latestInstallFailedA5.message) !== null && _latestInstallFailedA4 !== void 0 ? _latestInstallFailedA4 : '';
      }
      return {
        ...baseIntegrationData,
        install_status: {
          main: ccrIntegration.install_status,
          remote: localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status
        },
        updated_at: ccrIntegration.updated_at,
        sync_status: _types.SyncStatus.WARNING,
        ...((localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.latest_uninstall_failed_attempts) !== undefined ? {
          warning: {
            message: `${latestUninstallFailedAttempt} ${latestUninstallFailedAttemptTime}`,
            title: 'Integration was uninstalled, but removal from remote cluster failed.'
          }
        } : {})
      };
    }
    return {
      ...baseIntegrationData,
      install_status: {
        main: ccrIntegration.install_status,
        remote: localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status
      },
      sync_status: (localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.attributes.install_status) === 'installed' ? _types.SyncStatus.COMPLETED : _types.SyncStatus.SYNCHRONIZING,
      updated_at: localIntegrationSO === null || localIntegrationSO === void 0 ? void 0 : localIntegrationSO.updated_at
    };
  });
  return {
    integrations: integrationsStatus
  };
};
const fetchAndCompareCustomAssets = async (esClient, logger, ccrCustomAssets, installedIntegrationsByName) => {
  if (!ccrCustomAssets) return;
  const abortController = new AbortController();
  try {
    const installedPipelines = await (0, _custom_assets.getPipeline)(esClient, _custom_assets.CUSTOM_ASSETS_PREFIX, abortController);
    for (const [_, ccrCustomAsset] of Object.entries(ccrCustomAssets)) {
      if (ccrCustomAsset.type === 'ingest_pipeline' && !ccrCustomAsset.name.includes('@custom')) {
        const response = await (0, _custom_assets.getPipeline)(esClient, ccrCustomAsset.name, abortController);
        if (response[ccrCustomAsset.name]) {
          installedPipelines[ccrCustomAsset.name] = response[ccrCustomAsset.name];
        }
      }
    }
    const installedComponentTemplates = await (0, _custom_assets.getComponentTemplate)(esClient, _custom_assets.CUSTOM_ASSETS_PREFIX, abortController);
    const componentTemplatesByName = ((installedComponentTemplates === null || installedComponentTemplates === void 0 ? void 0 : installedComponentTemplates.component_templates) || []).reduce((acc, componentTemplate) => {
      if (componentTemplate !== null && componentTemplate !== void 0 && componentTemplate.name) {
        var _componentTemplate$co;
        acc[componentTemplate.name] = (_componentTemplate$co = componentTemplate.component_template) === null || _componentTemplate$co === void 0 ? void 0 : _componentTemplate$co.template;
      }
      return acc;
    }, {});
    const componentTemplates = installedComponentTemplates !== null && installedComponentTemplates !== void 0 && installedComponentTemplates.component_templates ? componentTemplatesByName : undefined;
    const result = {};

    // compare custom pipelines and custom component templates
    Object.entries(ccrCustomAssets).forEach(([ccrCustomName, ccrCustomAsset]) => {
      const res = compareCustomAssets({
        ccrCustomAsset,
        ingestPipelines: installedPipelines,
        componentTemplates,
        installedIntegration: installedIntegrationsByName[ccrCustomAsset.package_name]
      });
      result[ccrCustomName] = res;
    });
    return result;
  } catch (error) {
    logger.error('error', error === null || error === void 0 ? void 0 : error.message);
    return {
      error: error === null || error === void 0 ? void 0 : error.message
    };
  }
};
const compareCustomAssets = ({
  ccrCustomAsset,
  ingestPipelines,
  componentTemplates,
  installedIntegration
}) => {
  var _installedIntegration, _installedIntegration2, _latestCustomAssetErr;
  const result = {
    name: ccrCustomAsset.name,
    type: ccrCustomAsset.type,
    package_name: ccrCustomAsset.package_name,
    package_version: ccrCustomAsset.package_version
  };
  const latestCustomAssetError = installedIntegration === null || installedIntegration === void 0 ? void 0 : (_installedIntegration = installedIntegration.attributes) === null || _installedIntegration === void 0 ? void 0 : (_installedIntegration2 = _installedIntegration.latest_custom_asset_install_failed_attempts) === null || _installedIntegration2 === void 0 ? void 0 : _installedIntegration2[`${ccrCustomAsset.type}:${ccrCustomAsset.name}`];
  const latestFailedAttemptTime = latestCustomAssetError !== null && latestCustomAssetError !== void 0 && latestCustomAssetError.created_at ? `at ${new Date(latestCustomAssetError === null || latestCustomAssetError === void 0 ? void 0 : latestCustomAssetError.created_at).toUTCString()}` : '';
  const latestFailedAttempt = latestCustomAssetError !== null && latestCustomAssetError !== void 0 && (_latestCustomAssetErr = latestCustomAssetError.error) !== null && _latestCustomAssetErr !== void 0 && _latestCustomAssetErr.message ? `- reason: ${latestCustomAssetError.error.message}` : '';
  const latestFailedErrorMessage = `Failed to update ${ccrCustomAsset.type.replaceAll('_', ' ')} ${ccrCustomAsset.name} ${latestFailedAttempt} ${latestFailedAttemptTime}`;
  if (ccrCustomAsset.type === 'ingest_pipeline') {
    const installedPipeline = ingestPipelines === null || ingestPipelines === void 0 ? void 0 : ingestPipelines[ccrCustomAsset.name];
    if (!installedPipeline) {
      if (ccrCustomAsset.is_deleted === true) {
        return {
          ...result,
          is_deleted: true,
          sync_status: _types.SyncStatus.COMPLETED
        };
      }
      if (latestCustomAssetError) {
        return {
          ...result,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    }
    if (ccrCustomAsset.is_deleted === true && installedPipeline) {
      if (latestCustomAssetError) {
        return {
          ...result,
          is_deleted: true,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        is_deleted: true,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    } else if (installedPipeline !== null && installedPipeline !== void 0 && installedPipeline.version && installedPipeline.version < ccrCustomAsset.pipeline.version) {
      if (latestCustomAssetError) {
        return {
          ...result,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    } else if ((0, _lodash.isEqual)(installedPipeline, ccrCustomAsset === null || ccrCustomAsset === void 0 ? void 0 : ccrCustomAsset.pipeline)) {
      return {
        ...result,
        sync_status: _types.SyncStatus.COMPLETED
      };
    } else {
      if (latestCustomAssetError) {
        return {
          ...result,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    }
  } else if (ccrCustomAsset.type === 'component_template') {
    const installedCompTemplate = componentTemplates === null || componentTemplates === void 0 ? void 0 : componentTemplates[ccrCustomAsset.name];
    if (!installedCompTemplate) {
      if (ccrCustomAsset.is_deleted === true) {
        return {
          ...result,
          is_deleted: true,
          sync_status: _types.SyncStatus.COMPLETED
        };
      }
      if (latestCustomAssetError) {
        return {
          ...result,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    }
    if (ccrCustomAsset.is_deleted === true && installedCompTemplate) {
      if (latestCustomAssetError) {
        return {
          ...result,
          is_deleted: true,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        is_deleted: true,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    } else if ((0, _lodash.isEqual)(installedCompTemplate, ccrCustomAsset === null || ccrCustomAsset === void 0 ? void 0 : ccrCustomAsset.template)) {
      return {
        ...result,
        sync_status: _types.SyncStatus.COMPLETED
      };
    } else {
      if (latestCustomAssetError) {
        return {
          ...result,
          sync_status: _types.SyncStatus.FAILED,
          error: latestFailedErrorMessage
        };
      }
      return {
        ...result,
        sync_status: _types.SyncStatus.SYNCHRONIZING
      };
    }
  }
  return {};
};
const getRemoteSyncedIntegrationsStatus = async (esClient, soClient) => {
  const logger = _services.appContextService.getLogger();
  if (!(0, _fleet_synced_integrations.canEnableSyncIntegrations)()) {
    return {
      integrations: []
    };
  }
  try {
    const followerIndexRes = await getFollowerIndexInfo(esClient, logger);
    if (followerIndexRes !== null && followerIndexRes !== void 0 && followerIndexRes.error || !(followerIndexRes !== null && followerIndexRes !== void 0 && followerIndexRes.info)) {
      return {
        error: followerIndexRes === null || followerIndexRes === void 0 ? void 0 : followerIndexRes.error,
        integrations: []
      };
    }
    const res = await fetchAndCompareSyncedIntegrations(esClient, soClient, followerIndexRes.info.follower_index, logger);
    return res;
  } catch (error) {
    return {
      error: error.message,
      integrations: []
    };
  }
};
exports.getRemoteSyncedIntegrationsStatus = getRemoteSyncedIntegrationsStatus;