"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.additionalSourcesSuggestions = additionalSourcesSuggestions;
exports.buildPartialMatcher = buildPartialMatcher;
exports.checkFunctionInvocationComplete = checkFunctionInvocationComplete;
exports.correctQuerySyntax = correctQuerySyntax;
exports.extractTypeFromASTArg = extractTypeFromASTArg;
exports.getBracketsToClose = getBracketsToClose;
exports.getCompatibleTypesToSuggestNext = getCompatibleTypesToSuggestNext;
exports.getExpressionPosition = void 0;
exports.getFieldsOrFunctionsSuggestions = getFieldsOrFunctionsSuggestions;
exports.getFunctionsToIgnoreForStats = getFunctionsToIgnoreForStats;
exports.getOverlapRange = getOverlapRange;
exports.getQueryForFields = getQueryForFields;
exports.getSourceSuggestions = getSourceSuggestions;
exports.getSourcesFromCommands = getSourcesFromCommands;
exports.getSuggestionsToRightOfOperatorExpression = getSuggestionsToRightOfOperatorExpression;
exports.getSupportedTypesForBinaryOperators = getSupportedTypesForBinaryOperators;
exports.getValidFunctionSignaturesForPreviousArgs = getValidFunctionSignaturesForPreviousArgs;
exports.getValidSignaturesAndTypesToSuggestNext = getValidSignaturesAndTypesToSuggestNext;
exports.handleFragment = handleFragment;
exports.isAggFunctionUsedAlready = isAggFunctionUsedAlready;
exports.isExpressionComplete = isExpressionComplete;
exports.isLiteralDateItem = isLiteralDateItem;
exports.pushItUpInTheList = pushItUpInTheList;
exports.specialIndicesToSuggestions = void 0;
exports.strictlyGetParamAtPosition = strictlyGetParamAtPosition;
exports.suggestForExpression = suggestForExpression;
var _i18n = require("@kbn/i18n");
var _esqlAst = require("@kbn/esql-ast");
var _esqlTypes = require("@kbn/esql-types");
var _lodash = require("lodash");
var _all_operators = require("../definitions/all_operators");
var _types = require("../definitions/types");
var _constants = require("../shared/constants");
var _esql_types = require("../shared/esql_types");
var _helpers = require("../shared/helpers");
var _complete_items = require("./complete_items");
var _factories = require("./factories");
var _metadata = require("./commands/metadata");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * This function returns a list of closing brackets that can be appended to
 * a partial query to make it valid.
 *
 * A known limitation of this is that is not aware of commas "," or pipes "|"
 * so it is not yet helpful on a multiple commands errors (a workaround is to pass each command here...)
 * @param text
 * @returns
 */
function getBracketsToClose(text) {
  const stack = [];
  const pairs = {
    '"""': '"""',
    '/*': '*/',
    '(': ')',
    '[': ']',
    '"': '"'
  };
  const pairsReversed = {
    '"""': '"""',
    '*/': '/*',
    ')': '(',
    ']': '[',
    '"': '"'
  };
  for (let i = 0; i < text.length; i++) {
    for (const openBracket in pairs) {
      if (!Object.hasOwn(pairs, openBracket)) {
        continue;
      }
      const substr = text.slice(i, i + openBracket.length);
      if (pairsReversed[substr] && pairsReversed[substr] === stack[stack.length - 1]) {
        stack.pop();
        break;
      } else if (substr === openBracket) {
        stack.push(substr);
        break;
      }
    }
  }
  return stack.reverse().map(bracket => pairs[bracket]);
}

/**
 * This function attempts to correct the syntax of a partial query to make it valid.
 *
 * We are generally dealing with incomplete queries when the user is typing. But,
 * having an AST is helpful so we heuristically correct the syntax so it can be parsed.
 *
 * @param _query
 * @param context
 * @returns
 */
function correctQuerySyntax(_query) {
  let query = _query;
  // check if all brackets are closed, otherwise close them
  const bracketsToAppend = getBracketsToClose(query);
  const endsWithBinaryOperatorRegex = /(?:\+|\/|==|>=|>|in|<=|<|like|:|%|\*|-|not in|not like|not rlike|!=|rlike|and|or|not|=|as)\s+$/i;
  const endsWithCommaRegex = /,\s+$/;
  if (endsWithBinaryOperatorRegex.test(query) || endsWithCommaRegex.test(query)) {
    query += ` ${_constants.EDITOR_MARKER}`;
  }
  query += bracketsToAppend.join('');
  return query;
}
function extractFunctionArgs(args) {
  return args.flatMap(arg => (0, _helpers.isAssignment)(arg) ? arg.args[1] : arg).filter(_helpers.isFunctionItem);
}
function checkContent(fn) {
  const fnDef = (0, _helpers.getFunctionDefinition)(fn.name);
  return !!fnDef && fnDef.type === _types.FunctionDefinitionTypes.AGG || extractFunctionArgs(fn.args).some(checkContent);
}
function isAggFunctionUsedAlready(command, argIndex) {
  if (argIndex < 0) {
    return false;
  }
  const arg = command.args[argIndex];
  return (0, _helpers.isFunctionItem)(arg) ? checkContent(arg) : false;
}
function getFnContent(fn) {
  return [fn.name].concat(extractFunctionArgs(fn.args).flatMap(getFnContent));
}
function getFunctionsToIgnoreForStats(command, argIndex) {
  if (argIndex < 0) {
    return [];
  }
  const arg = command.args[argIndex];
  return (0, _helpers.isFunctionItem)(arg) ? getFnContent(arg) : [];
}

/**
 * Given a function signature, returns the parameter at the given position, even if it's undefined or null
 *
 * @param {params}
 * @param position
 * @returns
 */
function strictlyGetParamAtPosition({
  params
}, position) {
  return params[position] ? params[position] : null;
}

/**
 * This function is used to build the query that will be used to compute the
 * available fields for the current cursor location.
 *
 * Generally, this is the user's query up to the end of the previous command.
 *
 * @param queryString The original query string
 * @param commands
 * @returns
 */
function getQueryForFields(queryString, root) {
  const commands = root.commands;
  const lastCommand = commands[commands.length - 1];
  if (lastCommand && lastCommand.name === 'fork' && lastCommand.args.length > 0) {
    /**
     * This translates the current fork command branch into a simpler but equivalent
     * query that is compatible with the existing field computation/caching strategy.
     *
     * The intuition here is that if the cursor is within a fork branch, the
     * previous context is equivalent to a query without the FORK command.:
     *
     * Original query: FROM lolz | EVAL foo = 1 | FORK (EVAL bar = 2) (EVAL baz = 3 | WHERE /)
     * Simplified: FROM lolz | EVAL foo = 1 | EVAL baz = 3 | WHERE /
     */
    const currentBranch = lastCommand.args[lastCommand.args.length - 1];
    const newCommands = commands.slice(0, -1).concat(currentBranch.commands.slice(0, -1));
    return _esqlAst.BasicPrettyPrinter.print({
      ...root,
      commands: newCommands
    });
  }

  // If there is only one source command and it does not require fields, do not
  // fetch fields, hence return an empty string.
  return commands.length === 1 && ['row', 'show'].includes(commands[0].name) ? '' : buildQueryUntilPreviousCommand(queryString, commands);
}

// TODO consider replacing this with a pretty printer-based solution
function buildQueryUntilPreviousCommand(queryString, commands) {
  const prevCommand = commands[Math.max(commands.length - 2, 0)];
  return prevCommand ? queryString.substring(0, prevCommand.location.max + 1) : queryString;
}
function getSourcesFromCommands(commands, sourceType) {
  var _sourceCommand$args;
  const sourceCommand = commands.find(({
    name
  }) => name === 'from' || name === 'ts');
  const args = (_sourceCommand$args = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args !== void 0 ? _sourceCommand$args : [];
  // the marker gets added in queries like "FROM "
  return args.filter(arg => arg.sourceType === sourceType && arg.name !== '' && arg.name !== _constants.EDITOR_MARKER);
}
function getSupportedTypesForBinaryOperators(fnDef, previousType) {
  // Retrieve list of all 'right' supported types that match the left hand side of the function
  return fnDef && Array.isArray(fnDef === null || fnDef === void 0 ? void 0 : fnDef.signatures) ? fnDef.signatures.filter(({
    params
  }) => params.find(p => p.name === 'left' && p.type === previousType)).map(({
    params
  }) => params[1].type) : [previousType];
}
function getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex) {
  // Filter down to signatures that match every params up to the current argIndex
  // e.g. BUCKET(longField, /) => all signatures with first param as long column type
  // or BUCKET(longField, 2, /) => all signatures with (longField, integer, ...)
  const relevantFuncSignatures = fnDefinition.signatures.filter(s => {
    var _s$params;
    return ((_s$params = s.params) === null || _s$params === void 0 ? void 0 : _s$params.length) >= argIndex && s.params.slice(0, argIndex).every(({
      type: dataType
    }, idx) => {
      return dataType === enrichedArgs[idx].dataType || (0, _esql_types.compareTypesWithLiterals)(dataType, enrichedArgs[idx].dataType);
    });
  });
  return relevantFuncSignatures;
}

/**
 * Given a function signature, returns the compatible types to suggest for the next argument
 *
 * @param fnDefinition: the function definition
 * @param enrichedArgs: AST args with enriched esType info to match with function signatures
 * @param argIndex: the index of the argument to suggest for
 * @returns
 */
function getCompatibleTypesToSuggestNext(fnDefinition, enrichedArgs, argIndex) {
  // First, narrow down to valid function signatures based on previous arguments
  const relevantFuncSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);

  // Then, get the compatible types to suggest for the next argument
  const compatibleTypesToSuggestForArg = (0, _lodash.uniqBy)(relevantFuncSignatures.map(f => f.params[argIndex]).filter(d => d), o => `${o.type}-${o.constantOnly}`);
  return compatibleTypesToSuggestForArg;
}

/**
 * Checks the suggestion text for overlap with the current query.
 *
 * This is useful to determine the range of the existing query that should be
 * replaced if the suggestion is accepted.
 *
 * For example
 * QUERY: FROM source | WHERE field IS NO
 * SUGGESTION: IS NOT NULL
 *
 * The overlap is "IS NO" and the range to replace is "IS NO" in the query.
 *
 * @param query
 * @param suggestionText
 * @returns
 */
function getOverlapRange(query, suggestionText) {
  let overlapLength = 0;

  // Convert both strings to lowercase for case-insensitive comparison
  const lowerQuery = query.toLowerCase();
  const lowerSuggestionText = suggestionText.toLowerCase();
  for (let i = 0; i <= lowerSuggestionText.length; i++) {
    const substr = lowerSuggestionText.substring(0, i);
    if (lowerQuery.endsWith(substr)) {
      overlapLength = i;
    }
  }
  if (overlapLength === 0) {
    return;
  }
  return {
    start: query.length - overlapLength,
    end: query.length
  };
}
function isValidDateString(dateString) {
  if (typeof dateString !== 'string') return false;
  const timestamp = Date.parse(dateString.replace(/\"/g, ''));
  return !isNaN(timestamp);
}

/**
 * Returns true is node is a valid literal that represents a date
 * either a system time parameter or a date string generated by date picker
 * @param dateString
 * @returns
 */
function isLiteralDateItem(nodeArg) {
  return (0, _helpers.isLiteralItem)(nodeArg) && (
  // If text is ?start or ?end, it's a system time parameter
  _factories.TIME_SYSTEM_PARAMS.includes(nodeArg.text) ||
  // Or if it's a string generated by date picker
  isValidDateString(nodeArg.value));
}
function getValidSignaturesAndTypesToSuggestNext(node, references, fnDefinition, fullText, offset) {
  const enrichedArgs = node.args.map(nodeArg => {
    let dataType = extractTypeFromASTArg(nodeArg, references);

    // For named system time parameters ?start and ?end, make sure it's compatiable
    if (isLiteralDateItem(nodeArg)) {
      dataType = 'date';
    }
    return {
      ...nodeArg,
      dataType
    };
  });

  // pick the type of the next arg
  const shouldGetNextArgument = node.text.includes(_constants.EDITOR_MARKER);
  let argIndex = Math.max(node.args.length, 0);
  if (!shouldGetNextArgument && argIndex) {
    argIndex -= 1;
  }
  const validSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);
  // Retrieve unique of types that are compatiable for the current arg
  const typesToSuggestNext = getCompatibleTypesToSuggestNext(fnDefinition, enrichedArgs, argIndex);
  const hasMoreMandatoryArgs = !validSignatures
  // Types available to suggest next after this argument is completed
  .map(signature => strictlyGetParamAtPosition(signature, argIndex + 1))
  // when a param is null, it means param is optional
  // If there's at least one param that is optional, then
  // no need to suggest comma
  .some(p => p === null || (p === null || p === void 0 ? void 0 : p.optional) === true);

  // Whether to prepend comma to suggestion string
  // E.g. if true, "fieldName" -> "fieldName, "
  const alreadyHasComma = fullText ? fullText[offset] === ',' : false;
  const shouldAddComma = hasMoreMandatoryArgs && fnDefinition.type !== _types.FunctionDefinitionTypes.OPERATOR && !alreadyHasComma;
  const currentArg = enrichedArgs[argIndex];
  return {
    shouldAddComma,
    typesToSuggestNext,
    validSignatures,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex,
    currentArg
  };
}

/**
 * This function handles the logic to suggest completions
 * for a given fragment of text in a generic way. A good example is
 * a field name.
 *
 * When typing a field name, there are 2 scenarios
 *
 * 1. field name is incomplete (includes the empty string)
 * KEEP /
 * KEEP fie/
 *
 * 2. field name is complete
 * KEEP field/
 *
 * This function provides a framework for detecting and handling both scenarios in a clean way.
 *
 * @param innerText - the query text before the current cursor position
 * @param isFragmentComplete — return true if the fragment is complete
 * @param getSuggestionsForIncomplete — gets suggestions for an incomplete fragment
 * @param getSuggestionsForComplete - gets suggestions for a complete fragment
 * @returns
 */
function handleFragment(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete) {
  /**
   * @TODO — this string manipulation is crude and can't support all cases
   * Checking for a partial word and computing the replacement range should
   * really be done using the AST node, but we'll have to refactor further upstream
   * to make that available. This is a quick fix to support the most common case.
   */
  const fragment = (0, _helpers.findFinalWord)(innerText);
  if (!fragment) {
    return getSuggestionsForIncomplete('');
  } else {
    const rangeToReplace = {
      start: innerText.length - fragment.length,
      end: innerText.length
    };
    if (isFragmentComplete(fragment)) {
      return getSuggestionsForComplete(fragment, rangeToReplace);
    } else {
      return getSuggestionsForIncomplete(fragment, rangeToReplace);
    }
  }
}
/**
 * TODO — split this into distinct functions, one for fields, one for functions, one for literals
 */
async function getFieldsOrFunctionsSuggestions(types, location, getFieldsByType, {
  functions,
  fields,
  userDefinedColumns,
  values = false,
  literals = false
}, {
  ignoreFn = [],
  ignoreColumns = []
} = {}) {
  const filteredFieldsByType = pushItUpInTheList(await (fields ? getFieldsByType(types, ignoreColumns, {
    advanceCursor: location === _types.Location.SORT,
    openSuggestions: location === _types.Location.SORT,
    variableType: values ? _esqlTypes.ESQLVariableType.VALUES : _esqlTypes.ESQLVariableType.FIELDS
  }) : []), functions);
  const filteredColumnByType = [];
  if (userDefinedColumns) {
    for (const userDefinedColumn of userDefinedColumns.values()) {
      if ((types.includes('any') || types.includes(userDefinedColumn[0].type)) && !ignoreColumns.includes(userDefinedColumn[0].name)) {
        filteredColumnByType.push(userDefinedColumn[0].name);
      }
    }
    // due to a bug on the ES|QL table side, filter out fields list with underscored userDefinedColumns names (??)
    // avg( numberField ) => avg_numberField_
    const ALPHANUMERIC_REGEXP = /[^a-zA-Z\d]/g;
    if (filteredColumnByType.length && filteredColumnByType.some(v => ALPHANUMERIC_REGEXP.test(v))) {
      for (const userDefinedColumn of filteredColumnByType) {
        const underscoredName = userDefinedColumn.replace(ALPHANUMERIC_REGEXP, '_');
        const index = filteredFieldsByType.findIndex(({
          label
        }) => underscoredName === label || `_${underscoredName}_` === label);
        if (index >= 0) {
          filteredFieldsByType.splice(index);
        }
      }
    }
  }
  // could also be in stats (bucket) but our autocomplete is not great yet
  const displayDateSuggestions = types.includes('date') && [_types.Location.WHERE, _types.Location.EVAL].includes(location);
  const suggestions = filteredFieldsByType.concat(displayDateSuggestions ? (0, _factories.getDateLiterals)() : [], functions ? (0, _factories.getFunctionSuggestions)({
    location,
    returnTypes: types,
    ignored: ignoreFn
  }) : [], userDefinedColumns ? pushItUpInTheList((0, _factories.buildUserDefinedColumnsDefinitions)(filteredColumnByType), functions) : [], literals ? (0, _factories.getCompatibleLiterals)(types) : []);
  return suggestions;
}
function pushItUpInTheList(suggestions, shouldPromote) {
  if (!shouldPromote) {
    return suggestions;
  }
  return suggestions.map(({
    sortText,
    ...rest
  }) => ({
    ...rest,
    sortText: `1${sortText}`
  }));
}

/** @deprecated — use getExpressionType instead (src/platform/packages/shared/kbn-esql-validation-autocomplete/src/shared/helpers.ts) */
function extractTypeFromASTArg(arg, references) {
  if (Array.isArray(arg)) {
    return extractTypeFromASTArg(arg[0], references);
  }
  if ((0, _helpers.isLiteralItem)(arg)) {
    return arg.literalType;
  }
  if ((0, _helpers.isColumnItem)(arg) || (0, _esqlAst.isIdentifier)(arg)) {
    const hit = (0, _helpers.getColumnForASTNode)(arg, references);
    if (hit) {
      return hit.type;
    }
  }
  if ((0, _helpers.isTimeIntervalItem)(arg)) {
    return arg.type;
  }
  if ((0, _helpers.isFunctionItem)(arg)) {
    const fnDef = (0, _helpers.getFunctionDefinition)(arg.name);
    if (fnDef) {
      // @TODO: improve this to better filter down the correct return type based on existing arguments
      // just mind that this can be highly recursive...
      return fnDef.signatures[0].returnType;
    }
  }
}

/**
 * In several cases we don't want to count the last arg if it is
 * of type unknown.
 *
 * this solves for the case where the user has typed a
 * prefix (e.g. "keywordField != tex/")
 *
 * "tex" is not a recognizable identifier so it is of
 * type "unknown" which leads us to continue suggesting
 * fields/functions.
 *
 * Monaco will then filter our suggestions list
 * based on the "tex" prefix which gives the correct UX
 */
function removeFinalUnknownIdentiferArg(args, getExpressionType) {
  return getExpressionType(args[args.length - 1]) === 'unknown' ? args.slice(0, args.length - 1) : args;
}

// @TODO: refactor this to be shared with validation
function checkFunctionInvocationComplete(func, getExpressionType) {
  const fnDefinition = (0, _helpers.getFunctionDefinition)(func.name);
  if (!fnDefinition) {
    return {
      complete: false
    };
  }
  const cleanedArgs = removeFinalUnknownIdentiferArg(func.args, getExpressionType);
  const argLengthCheck = fnDefinition.signatures.some(def => {
    if (def.minParams && cleanedArgs.length >= def.minParams) {
      return true;
    }
    if (cleanedArgs.length === def.params.length) {
      return true;
    }
    return cleanedArgs.length >= def.params.filter(({
      optional
    }) => !optional).length;
  });
  if (!argLengthCheck) {
    return {
      complete: false,
      reason: 'tooFewArgs'
    };
  }
  if ((fnDefinition.name === 'in' || fnDefinition.name === 'not in') && Array.isArray(func.args[1]) && !func.args[1].length) {
    return {
      complete: false,
      reason: 'tooFewArgs'
    };
  }

  // If the function is complete, check that the types of the arguments match the function definition
  const hasCorrectTypes = fnDefinition.signatures.some(def => {
    return func.args.every((a, index) => {
      return fnDefinition.name.endsWith('null') || def.params[index].type === 'any' || def.params[index].type === getExpressionType(a) ||
      // this is a special case for expressions with named parameters
      // e.g. "WHERE field == ?value"
      (0, _helpers.isParamExpressionType)(getExpressionType(a));
    });
  });
  if (!hasCorrectTypes) {
    return {
      complete: false,
      reason: 'wrongTypes'
    };
  }
  return {
    complete: true
  };
}

/**
 * This function is used to
 * - suggest the next argument for an incomplete or incorrect binary operator expression (e.g. field > <suggest>)
 * - suggest an operator to the right of a complete binary operator expression (e.g. field > 0 <suggest>)
 * - suggest an operator to the right of a complete unary operator (e.g. field IS NOT NULL <suggest>)
 *
 * TODO — is this function doing too much?
 */
async function getSuggestionsToRightOfOperatorExpression({
  queryText,
  location,
  rootOperator: operator,
  preferredExpressionType,
  getExpressionType,
  getColumnsByType
}) {
  const suggestions = [];
  const isFnComplete = checkFunctionInvocationComplete(operator, getExpressionType);
  if (isFnComplete.complete) {
    // i.e. ... | <COMMAND> field > 0 <suggest>
    // i.e. ... | <COMMAND> field + otherN <suggest>
    const operatorReturnType = getExpressionType(operator);
    suggestions.push(...(0, _factories.getOperatorSuggestions)({
      location,
      // here we use the operator return type because we're suggesting operators that could
      // accept the result of the existing operator as a left operand
      leftParamType: operatorReturnType === 'unknown' || operatorReturnType === 'unsupported' ? 'any' : operatorReturnType,
      ignored: ['=', ':']
    }));
  } else {
    // i.e. ... | <COMMAND> field >= <suggest>
    // i.e. ... | <COMMAND> field + <suggest>
    // i.e. ... | <COMMAND> field and <suggest>

    // Because it's an incomplete function, need to extract the type of the current argument
    // and suggest the next argument based on types

    // pick the last arg and check its type to verify whether is incomplete for the given function
    const cleanedArgs = removeFinalUnknownIdentiferArg(operator.args, getExpressionType);
    const leftArgType = getExpressionType(operator.args[cleanedArgs.length - 1]);
    if (isFnComplete.reason === 'tooFewArgs') {
      const fnDef = (0, _helpers.getFunctionDefinition)(operator.name);
      if (fnDef !== null && fnDef !== void 0 && fnDef.signatures.every(({
        params
      }) => params.some(({
        type
      }) => (0, _helpers.isArrayType)(type)))) {
        suggestions.push(_complete_items.listCompleteItem);
      } else {
        var _getFunctionDefinitio;
        const finalType = leftArgType || 'any';
        const supportedTypes = getSupportedTypesForBinaryOperators(fnDef, finalType);

        // this is a special case with AND/OR
        // <COMMAND> expression AND/OR <suggest>
        // technically another boolean value should be suggested, but it is a better experience
        // to actually suggest a wider set of fields/functions
        const typeToUse = finalType === 'boolean' && ((_getFunctionDefinitio = (0, _helpers.getFunctionDefinition)(operator.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.OPERATOR ? ['any'] : supportedTypes;

        // TODO replace with fields callback + function suggestions
        suggestions.push(...(await getFieldsOrFunctionsSuggestions(typeToUse, location, getColumnsByType, {
          functions: true,
          fields: true,
          values: Boolean(operator.subtype === 'binary-expression')
        })));
      }
    }

    /**
     * If the caller has supplied a preferred expression type, we can suggest operators that
     * would move the user toward that expression type.
     *
     * e.g. if we have a preferred type of boolean and we have `timestamp > "2002" AND doubleField`
     * this is an incorrect signature for AND because the left side is boolean and the right side is double
     *
     * Knowing that we prefer boolean expressions, we suggest operators that would accept doubleField as a left operand
     * and also return a boolean value.
     *
     * I believe this is only used in WHERE and probably bears some rethinking.
     */
    if (isFnComplete.reason === 'wrongTypes') {
      if (leftArgType && preferredExpressionType) {
        // suggest something to complete the operator
        if (leftArgType !== preferredExpressionType && (0, _types.isParameterType)(leftArgType) && (0, _types.isReturnType)(preferredExpressionType)) {
          suggestions.push(...(0, _factories.getOperatorSuggestions)({
            location,
            leftParamType: leftArgType,
            returnTypes: [preferredExpressionType]
          }));
        }
      }
    }
  }
  return suggestions.map(s => {
    const overlap = getOverlapRange(queryText, s.text);
    return {
      ...s,
      rangeToReplace: overlap
    };
  });
}

/**
 * The position of the cursor within an expression.
 */

/**
 * Escapes special characters in a string to be used as a literal match in a regular expression.
 * @param {string} text The input string to escape.
 * @returns {string} The escaped string.
 */
function escapeRegExp(text) {
  // Characters with special meaning in regex: . * + ? ^ $ { } ( ) | [ ] \
  // We need to escape all of them. The `$&` in the replacement string means "the matched substring".
  return text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

/**
 * Determines the position of the cursor within an expression.
 * @param innerText
 * @param expressionRoot
 * @returns
 */
const getExpressionPosition = (innerText, expressionRoot) => {
  const endsWithNot = / not$/i.test(innerText.trimEnd());
  if (endsWithNot && !(expressionRoot && (0, _helpers.isFunctionItem)(expressionRoot) &&
  // See https://github.com/elastic/kibana/issues/199401
  // for more information on this check...
  ['is null', 'is not null'].includes(expressionRoot.name))) {
    return 'after_not';
  }
  if (expressionRoot) {
    if ((0, _helpers.isColumnItem)(expressionRoot) &&
    // and not directly after the column name or prefix e.g. "colu/"
    // we are escaping the column name here as it may contain special characters such as ??
    !new RegExp(`${escapeRegExp(expressionRoot.parts.join('\\.'))}$`).test(innerText)) {
      return 'after_column';
    }
    if ((0, _helpers.isFunctionItem)(expressionRoot) && expressionRoot.subtype === 'variadic-call') {
      return 'after_function';
    }
    if ((0, _helpers.isFunctionItem)(expressionRoot) && expressionRoot.subtype !== 'variadic-call') {
      return 'after_operator';
    }
    if ((0, _helpers.isLiteralItem)(expressionRoot) || (0, _helpers.isTimeIntervalItem)(expressionRoot)) {
      return 'after_literal';
    }
  }
  return 'empty_expression';
};

/**
 * Creates suggestion within an expression.
 *
 * TODO — should this function know about the command context
 * or would we prefer a set of generic configuration options?
 *
 * @param param0
 * @returns
 */
exports.getExpressionPosition = getExpressionPosition;
async function suggestForExpression({
  expressionRoot,
  innerText,
  getExpressionType,
  getColumnsByType,
  previousCommands,
  location,
  preferredExpressionType
}) {
  var _previousCommands$map, _last$text;
  const suggestions = [];
  const position = getExpressionPosition(innerText, expressionRoot);
  switch (position) {
    /**
     * After a literal, column, or complete (non-operator) function call
     */
    case 'after_literal':
    case 'after_column':
    case 'after_function':
      const expressionType = getExpressionType(expressionRoot);
      if (!(0, _types.isParameterType)(expressionType)) {
        break;
      }
      suggestions.push(...(0, _factories.getOperatorSuggestions)({
        location,
        // In case of a param literal, we don't know the type of the left operand
        // so we can only suggest operators that accept any type as a left operand
        leftParamType: (0, _helpers.isParamExpressionType)(expressionType) ? undefined : expressionType,
        ignored: ['=']
      }));
      break;

    /**
     * After a NOT keyword
     *
     * the NOT function is a special operator that can be used in different ways,
     * and not all these are mapped within the AST data structure: in particular
     * <COMMAND> <field> NOT <here>
     * is an incomplete statement and it results in a missing AST node, so we need to detect
     * from the query string itself
     *
     * (this comment was copied but seems to still apply)
     */
    case 'after_not':
      if (expressionRoot && (0, _helpers.isFunctionItem)(expressionRoot) && expressionRoot.name === 'not') {
        suggestions.push(...(0, _factories.getFunctionSuggestions)({
          location,
          returnTypes: ['boolean']
        }), ...(await getColumnsByType('boolean', [], {
          advanceCursor: true,
          openSuggestions: true
        })));
      } else {
        suggestions.push(...(0, _factories.getSuggestionsAfterNot)());
      }
      break;

    /**
     * After an operator (e.g. AND, OR, IS NULL, +, etc.)
     */
    case 'after_operator':
      if (!expressionRoot) {
        break;
      }
      if (!(0, _helpers.isFunctionItem)(expressionRoot) || expressionRoot.subtype === 'variadic-call') {
        // this is already guaranteed in the getPosition function, but TypeScript doesn't know
        break;
      }
      let rightmostOperator = expressionRoot;
      // get rightmost function
      const walker = new _esqlAst.Walker({
        visitFunction: fn => {
          if (fn.location.min > rightmostOperator.location.min && fn.subtype !== 'variadic-call') rightmostOperator = fn;
        }
      });
      walker.walkFunction(expressionRoot);

      // See https://github.com/elastic/kibana/issues/199401 for an explanation of
      // why this check has to be so convoluted
      if (rightmostOperator.text.toLowerCase().trim().endsWith('null')) {
        suggestions.push(..._all_operators.logicalOperators.map(_factories.getOperatorSuggestion));
        break;
      }
      suggestions.push(...(await getSuggestionsToRightOfOperatorExpression({
        queryText: innerText,
        location,
        rootOperator: rightmostOperator,
        preferredExpressionType,
        getExpressionType,
        getColumnsByType
      })));
      break;
    case 'empty_expression':
      // Don't suggest MATCH, QSTR or KQL after unsupported commands
      const priorCommands = (_previousCommands$map = previousCommands === null || previousCommands === void 0 ? void 0 : previousCommands.map(a => a.name)) !== null && _previousCommands$map !== void 0 ? _previousCommands$map : [];
      const ignored = [];
      if (priorCommands.some(c => _constants.UNSUPPORTED_COMMANDS_BEFORE_MATCH.has(c))) {
        ignored.push('match');
      }
      if (priorCommands.some(c => _constants.UNSUPPORTED_COMMANDS_BEFORE_QSTR.has(c))) {
        ignored.push('kql', 'qstr');
      }
      const last = previousCommands === null || previousCommands === void 0 ? void 0 : previousCommands[previousCommands.length - 1];
      let columnSuggestions = [];
      if (!(last !== null && last !== void 0 && (_last$text = last.text) !== null && _last$text !== void 0 && _last$text.endsWith(`:${_constants.EDITOR_MARKER}`))) {
        columnSuggestions = await getColumnsByType('any', [], {
          advanceCursor: true,
          openSuggestions: true
        });
      }
      suggestions.push(...pushItUpInTheList(columnSuggestions, true), ...(0, _factories.getFunctionSuggestions)({
        location,
        ignored
      }));
      break;
  }

  /**
   * Attach replacement ranges if there's a prefix.
   *
   * Can't rely on Monaco because
   * - it counts "." as a word separator
   * - it doesn't handle multi-word completions (like "is null")
   *
   * TODO - think about how to generalize this — issue: https://github.com/elastic/kibana/issues/209905
   */
  const hasNonWhitespacePrefix = !/\s/.test(innerText[innerText.length - 1]);
  suggestions.forEach(s => {
    if (['IS NULL', 'IS NOT NULL'].includes(s.text)) {
      // this suggestion has spaces in it (e.g. "IS NOT NULL")
      // so we need to see if there's an overlap
      s.rangeToReplace = getOverlapRange(innerText, s.text);
      return;
    } else if (hasNonWhitespacePrefix) {
      // get index of first char of final word
      const lastNonWhitespaceIndex = innerText.search(/\S(?=\S*$)/);
      s.rangeToReplace = {
        start: lastNonWhitespaceIndex,
        end: innerText.length
      };
    }
  });
  return suggestions;
}

/**
 * Builds a regex that matches partial strings starting
 * from the beginning of the string.
 *
 * Example:
 * "is null" -> /^i(?:s(?:\s+(?:n(?:u(?:l(?:l)?)?)?)?)?)?$/i
 */
function buildPartialMatcher(str) {
  // Split the string into characters
  const chars = str.split('');

  // Initialize the regex pattern
  let pattern = '';

  // Iterate through the characters and build the pattern
  chars.forEach((char, index) => {
    if (char === ' ') {
      pattern += '\\s+';
    } else {
      pattern += char;
    }
    if (index < chars.length - 1) {
      pattern += '(?:';
    }
  });

  // Close the non-capturing groups
  for (let i = 0; i < chars.length - 1; i++) {
    pattern += ')?';
  }

  // Return the final regex pattern
  return new RegExp(pattern + '$', 'i');
}
const isNullMatcher = buildPartialMatcher('is nul');
const isNotNullMatcher = buildPartialMatcher('is not nul');

/**
 * Checks whether an expression is truly complete.
 *
 * (Encapsulates handling of the "is null" and "is not null"
 * checks)
 *
 * @todo use the simpler "getExpressionType(root) !== 'unknown'"
 * as soon as https://github.com/elastic/kibana/issues/199401 is resolved
 */
function isExpressionComplete(expressionType, innerText) {
  return expressionType !== 'unknown' &&
  // see https://github.com/elastic/kibana/issues/199401
  // for the reason we need this string check.
  !(isNullMatcher.test(innerText) || isNotNullMatcher.test(innerText));
}
function getSourceSuggestions(sources, alreadyUsed) {
  // hide indexes that start with .
  return (0, _factories.buildSourcesDefinitions)(sources.filter(({
    hidden,
    name
  }) => !hidden && !alreadyUsed.includes(name)).map(({
    name,
    dataStreams,
    title,
    type
  }) => {
    return {
      name,
      isIntegration: Boolean(dataStreams && dataStreams.length),
      title,
      type
    };
  }));
}
async function additionalSourcesSuggestions(queryText, sources, ignored, recommendedQuerySuggestions) {
  const suggestionsToAdd = await handleFragment(queryText, fragment => (0, _helpers.sourceExists)(fragment, new Set(sources.map(({
    name: sourceName
  }) => sourceName))), (_fragment, rangeToReplace) => {
    return getSourceSuggestions(sources, ignored).map(suggestion => ({
      ...suggestion,
      rangeToReplace
    }));
  }, (fragment, rangeToReplace) => {
    const exactMatch = sources.find(({
      name: _name
    }) => _name === fragment);
    if (exactMatch !== null && exactMatch !== void 0 && exactMatch.dataStreams) {
      // this is an integration name, suggest the datastreams
      const definitions = (0, _factories.buildSourcesDefinitions)(exactMatch.dataStreams.map(({
        name
      }) => ({
        name,
        isIntegration: false
      })));
      return definitions;
    } else {
      const _suggestions = [{
        ..._complete_items.pipeCompleteItem,
        filterText: fragment,
        text: fragment + ' | ',
        command: _factories.TRIGGER_SUGGESTION_COMMAND,
        rangeToReplace
      }, {
        ..._complete_items.commaCompleteItem,
        filterText: fragment,
        text: fragment + ', ',
        command: _factories.TRIGGER_SUGGESTION_COMMAND,
        rangeToReplace
      }, {
        ..._metadata.metadataSuggestion,
        filterText: fragment,
        text: fragment + ' METADATA ',
        rangeToReplace
      }, ...recommendedQuerySuggestions.map(suggestion => ({
        ...suggestion,
        rangeToReplace,
        filterText: fragment,
        text: fragment + suggestion.text
      }))];
      return _suggestions;
    }
  });
  return suggestionsToAdd;
}

// Treating lookup and time_series mode indices
const specialIndicesToSuggestions = indices => {
  const mainSuggestions = [];
  const aliasSuggestions = [];
  for (const index of indices) {
    mainSuggestions.push({
      label: index.name,
      text: index.name + ' ',
      kind: 'Issue',
      detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.specialIndexes.indexType.index', {
        defaultMessage: 'Index'
      }),
      sortText: '0-INDEX-' + index.name,
      command: _factories.TRIGGER_SUGGESTION_COMMAND
    });
    if (index.aliases) {
      for (const alias of index.aliases) {
        aliasSuggestions.push({
          label: alias,
          text: alias + ' $0',
          kind: 'Issue',
          detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.specialIndexes.indexType.alias', {
            defaultMessage: 'Alias'
          }),
          sortText: '1-ALIAS-' + alias,
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        });
      }
    }
  }
  return [...mainSuggestions, ...aliasSuggestions];
};
exports.specialIndicesToSuggestions = specialIndicesToSuggestions;