"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchResultProvider = getSearchResultProvider;
exports.toSearchResult = toSearchResult;
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function toSearchResult({
  iconPath,
  name,
  score,
  serviceType,
  url
}) {
  const newUrl = `${_constants.ENTERPRISE_SEARCH_DATA_PLUGIN.URL}/connectors/select_connector`;
  return {
    icon: iconPath || 'logoElasticsearch',
    id: serviceType,
    score,
    title: name,
    type: 'Elasticsearch',
    url: {
      path: url !== null && url !== void 0 ? url : newUrl,
      prependBasePath: true
    }
  };
}
function getSearchResultProvider(config, connectorTypes) {
  return {
    find: ({
      term,
      types,
      tags
    }, {
      aborted$,
      maxResults
    }, {
      core: {
        capabilities
      }
    }) => {
      if (tags || types && !(types.includes('integration') || types.includes('enterprise search'))) {
        return (0, _rxjs.of)([]);
      }
      return capabilities.pipe((0, _rxjs.takeUntil)(aborted$), (0, _rxjs.map)(caps => {
        if (!caps.catalogue.enterpriseSearch) {
          return [];
        }
        const services = [...(config.hasConnectors ? connectorTypes : []), {
          keywords: ['esre', 'search'],
          name: _i18n.i18n.translate('xpack.enterpriseSearch.searchProvider.aiSearch.name', {
            defaultMessage: 'Search AI'
          }),
          serviceType: 'ai_search',
          url: _constants.AI_SEARCH_PLUGIN.URL
        }];
        const result = services.map(service => {
          const {
            iconPath,
            name,
            keywords,
            serviceType
          } = service;
          const url = 'url' in service ? service.url : undefined;
          let score = 0;
          const searchTerm = (term || '').toLowerCase();
          const searchName = name.toLowerCase();
          if (!searchTerm) {
            score = 80;
          } else if (searchName === searchTerm) {
            score = 100;
          } else if (searchName.startsWith(searchTerm)) {
            score = 90;
          } else if (searchName.includes(searchTerm)) {
            score = 75;
          } else if (serviceType === searchTerm) {
            score = 65;
          } else if (keywords.some(keyword => keyword.includes(searchTerm))) {
            score = 50;
          }
          return toSearchResult({
            iconPath,
            name,
            score,
            serviceType,
            url
          });
        }).filter(({
          score
        }) => score > 0).slice(0, maxResults);
        return result;
      }));
    },
    getSearchableTypes: () => ['enterprise search', 'integration'],
    id: 'enterpriseSearch'
  };
}