"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useServiceMap = void 0;
var _react = require("react");
var _use_apm_plugin_context = require("../../../context/apm_plugin/use_apm_plugin_context");
var _use_license_context = require("../../../context/license/use_license_context");
var _license_check = require("../../../../common/license_check");
var _service_map = require("../../../../common/service_map");
var _use_fetcher = require("../../../hooks/use_fetcher");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INITIAL_SERVICE_MAP_STATE = {
  elements: [],
  nodesCount: 0,
  tracesCount: 0
};
const useServiceMap = ({
  start,
  end,
  environment,
  serviceName,
  serviceGroupId,
  kuery,
  isServiceMapApiV2Enabled
}) => {
  const license = (0, _use_license_context.useLicenseContext)();
  const {
    config
  } = (0, _use_apm_plugin_context.useApmPluginContext)();
  const [serviceMapNodes, setServiceMapNodes] = (0, _react.useState)({
    data: INITIAL_SERVICE_MAP_STATE,
    status: _use_fetcher.FETCH_STATUS.LOADING
  });
  const {
    data,
    status,
    error
  } = (0, _use_fetcher.useFetcher)(callApmApi => {
    // When we don't have a license or a valid license, don't make the request.
    if (!license || !(0, _license_check.isActivePlatinumLicense)(license) || !config.serviceMapEnabled) {
      return;
    }
    return callApmApi('GET /internal/apm/service-map', {
      params: {
        query: {
          start,
          end,
          environment,
          serviceName,
          serviceGroup: serviceGroupId,
          kuery,
          useV2: isServiceMapApiV2Enabled
        }
      }
    });
  }, [license, serviceName, environment, start, end, serviceGroupId, kuery, config.serviceMapEnabled, isServiceMapApiV2Enabled], {
    preservePreviousData: false
  });
  (0, _react.useEffect)(() => {
    if (status === _use_fetcher.FETCH_STATUS.LOADING) {
      setServiceMapNodes(prevState => ({
        ...prevState,
        status: _use_fetcher.FETCH_STATUS.LOADING
      }));
      return;
    }
    if (status === _use_fetcher.FETCH_STATUS.FAILURE || error) {
      setServiceMapNodes({
        data: INITIAL_SERVICE_MAP_STATE,
        status: _use_fetcher.FETCH_STATUS.FAILURE,
        error
      });
      return;
    }
    if (data) {
      if ('spans' in data) {
        try {
          const transformedData = processServiceMapData(data);
          setServiceMapNodes({
            data: {
              elements: transformedData.elements,
              nodesCount: transformedData.nodesCount,
              tracesCount: data.tracesCount
            },
            status: _use_fetcher.FETCH_STATUS.SUCCESS
          });
        } catch (err) {
          setServiceMapNodes({
            data: INITIAL_SERVICE_MAP_STATE,
            status: _use_fetcher.FETCH_STATUS.FAILURE,
            error: err
          });
        }
      } else {
        setServiceMapNodes({
          data,
          status: _use_fetcher.FETCH_STATUS.SUCCESS
        });
      }
    }
  }, [data, status, error]);
  return serviceMapNodes;
};
exports.useServiceMap = useServiceMap;
const processServiceMapData = data => {
  const paths = (0, _service_map.getPaths)({
    spans: data.spans
  });
  return (0, _service_map.getServiceMapNodes)({
    connections: paths.connections,
    exitSpanDestinations: paths.exitSpanDestinations,
    servicesData: data.servicesData,
    anomalies: data.anomalies
  });
};