"use strict";
/*
 * Copyright Elasticsearch B.V. and contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const commonQueryParams = ['error_trace', 'filter_path', 'human', 'pretty'];
class Snapshot {
    constructor(transport) {
        Object.defineProperty(this, "transport", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "acceptedParams", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.transport = transport;
        this.acceptedParams = {
            'snapshot.cleanup_repository': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'master_timeout',
                    'timeout'
                ]
            },
            'snapshot.clone': {
                path: [
                    'repository',
                    'snapshot',
                    'target_snapshot'
                ],
                body: [
                    'indices'
                ],
                query: [
                    'master_timeout'
                ]
            },
            'snapshot.create': {
                path: [
                    'repository',
                    'snapshot'
                ],
                body: [
                    'expand_wildcards',
                    'feature_states',
                    'ignore_unavailable',
                    'include_global_state',
                    'indices',
                    'metadata',
                    'partial'
                ],
                query: [
                    'master_timeout',
                    'wait_for_completion'
                ]
            },
            'snapshot.create_repository': {
                path: [
                    'name'
                ],
                body: [
                    'repository'
                ],
                query: [
                    'master_timeout',
                    'timeout',
                    'verify'
                ]
            },
            'snapshot.delete': {
                path: [
                    'repository',
                    'snapshot'
                ],
                body: [],
                query: [
                    'master_timeout',
                    'wait_for_completion'
                ]
            },
            'snapshot.delete_repository': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'master_timeout',
                    'timeout'
                ]
            },
            'snapshot.get': {
                path: [
                    'repository',
                    'snapshot'
                ],
                body: [],
                query: [
                    'after',
                    'from_sort_value',
                    'ignore_unavailable',
                    'index_details',
                    'index_names',
                    'include_repository',
                    'master_timeout',
                    'order',
                    'offset',
                    'size',
                    'slm_policy_filter',
                    'sort',
                    'state',
                    'verbose'
                ]
            },
            'snapshot.get_repository': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'local',
                    'master_timeout'
                ]
            },
            'snapshot.repository_analyze': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'blob_count',
                    'concurrency',
                    'detailed',
                    'early_read_node_count',
                    'max_blob_size',
                    'max_total_data_size',
                    'rare_action_probability',
                    'rarely_abort_writes',
                    'read_node_count',
                    'register_operation_count',
                    'seed',
                    'timeout'
                ]
            },
            'snapshot.repository_verify_integrity': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'blob_thread_pool_concurrency',
                    'index_snapshot_verification_concurrency',
                    'index_verification_concurrency',
                    'max_bytes_per_sec',
                    'max_failed_shard_snapshots',
                    'meta_thread_pool_concurrency',
                    'snapshot_verification_concurrency',
                    'verify_blob_contents'
                ]
            },
            'snapshot.restore': {
                path: [
                    'repository',
                    'snapshot'
                ],
                body: [
                    'feature_states',
                    'ignore_index_settings',
                    'ignore_unavailable',
                    'include_aliases',
                    'include_global_state',
                    'index_settings',
                    'indices',
                    'partial',
                    'rename_pattern',
                    'rename_replacement'
                ],
                query: [
                    'master_timeout',
                    'wait_for_completion'
                ]
            },
            'snapshot.status': {
                path: [
                    'repository',
                    'snapshot'
                ],
                body: [],
                query: [
                    'ignore_unavailable',
                    'master_timeout'
                ]
            },
            'snapshot.verify_repository': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'master_timeout',
                    'timeout'
                ]
            }
        };
    }
    async cleanupRepository(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.cleanup_repository'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_snapshot/${encodeURIComponent(params.name.toString())}/_cleanup`;
        const meta = {
            name: 'snapshot.cleanup_repository',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async clone(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['snapshot.clone'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/${encodeURIComponent(params.snapshot.toString())}/_clone/${encodeURIComponent(params.target_snapshot.toString())}`;
        const meta = {
            name: 'snapshot.clone',
            pathParts: {
                repository: params.repository,
                snapshot: params.snapshot,
                target_snapshot: params.target_snapshot
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async create(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['snapshot.create'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/${encodeURIComponent(params.snapshot.toString())}`;
        const meta = {
            name: 'snapshot.create',
            pathParts: {
                repository: params.repository,
                snapshot: params.snapshot
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async createRepository(params, options) {
        var _a;
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['snapshot.create_repository'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body = (_a = params.body) !== null && _a !== void 0 ? _a : undefined;
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                // @ts-expect-error
                body = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_snapshot/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'snapshot.create_repository',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async delete(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.delete'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/${encodeURIComponent(params.snapshot.toString())}`;
        const meta = {
            name: 'snapshot.delete',
            pathParts: {
                repository: params.repository,
                snapshot: params.snapshot
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async deleteRepository(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.delete_repository'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_snapshot/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'snapshot.delete_repository',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async get(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.get'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/${encodeURIComponent(params.snapshot.toString())}`;
        const meta = {
            name: 'snapshot.get',
            pathParts: {
                repository: params.repository,
                snapshot: params.snapshot
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getRepository(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.get_repository'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.name != null) {
            method = 'GET';
            path = `/_snapshot/${encodeURIComponent(params.name.toString())}`;
        }
        else {
            method = 'GET';
            path = '/_snapshot';
        }
        const meta = {
            name: 'snapshot.get_repository',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async repositoryAnalyze(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.repository_analyze'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_snapshot/${encodeURIComponent(params.name.toString())}/_analyze`;
        const meta = {
            name: 'snapshot.repository_analyze',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async repositoryVerifyIntegrity(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.repository_verify_integrity'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_snapshot/${encodeURIComponent(params.name.toString())}/_verify_integrity`;
        const meta = {
            name: 'snapshot.repository_verify_integrity',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async restore(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['snapshot.restore'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/${encodeURIComponent(params.snapshot.toString())}/_restore`;
        const meta = {
            name: 'snapshot.restore',
            pathParts: {
                repository: params.repository,
                snapshot: params.snapshot
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async status(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.status'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.repository != null && params.snapshot != null) {
            method = 'GET';
            path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/${encodeURIComponent(params.snapshot.toString())}/_status`;
        }
        else if (params.repository != null) {
            method = 'GET';
            path = `/_snapshot/${encodeURIComponent(params.repository.toString())}/_status`;
        }
        else {
            method = 'GET';
            path = '/_snapshot/_status';
        }
        const meta = {
            name: 'snapshot.status',
            pathParts: {
                repository: params.repository,
                snapshot: params.snapshot
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async verifyRepository(params, options) {
        const { path: acceptedPath } = this.acceptedParams['snapshot.verify_repository'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_snapshot/${encodeURIComponent(params.name.toString())}/_verify`;
        const meta = {
            name: 'snapshot.verify_repository',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
}
exports.default = Snapshot;
//# sourceMappingURL=snapshot.js.map