"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexExplorerTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexExplorerSchema = _zod.z.object({
  query: _zod.z.string().describe('A natural language query to infer which indices to use.'),
  limit: _zod.z.number().optional().describe('(optional) Limit the max number of indices to return. Defaults to 1.'),
  indexPattern: _zod.z.string().optional().describe('(optional) Index pattern to filter indices by. Defaults to *.')
});
const indexExplorerTool = () => {
  return {
    id: _onechatCommon.BuiltinToolIds.indexExplorer,
    description: `List relevant indices and corresponding mappings based on a natural language query.

                  The 'indexPattern' parameter can be used to filter indices by a specific pattern, e.g. 'foo*'.
                  This should *only* be used if you know what you're doing (e.g. if the user explicitly specified a pattern).
                  Otherwise, leave it empty to list all indices.

                  *Example:*
                  User: "Show me my latest alerts"
                  You: call tool 'indexExplorer' with { query: 'indices containing alerts' }
                  Tool result: [{ indexName: '.alerts', mappings: {...} }]
                  `,
    schema: indexExplorerSchema,
    handler: async ({
      query,
      indexPattern = '*',
      limit = 1
    }, {
      esClient,
      modelProvider
    }) => {
      const model = await modelProvider.getDefaultModel();
      return (0, _onechatGenaiUtils.indexExplorer)({
        query,
        indexPattern,
        limit,
        esClient: esClient.asCurrentUser,
        model
      });
    },
    meta: {
      tags: [_onechatCommon.BuiltinTags.retrieval]
    }
  };
};
exports.indexExplorerTool = indexExplorerTool;