"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAgentRoutes = registerAgentRoutes;
var _configSchema = require("@kbn/config-schema");
var _rxjs = require("rxjs");
var _sseUtilsServer = require("@kbn/sse-utils-server");
var _features = require("../../common/features");
var _wrap_handler = require("./wrap_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerAgentRoutes({
  router,
  getInternalServices,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });
  router.post({
    path: '/internal/onechat/agents/invoke',
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        agentId: _configSchema.schema.string({}),
        agentParams: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
      })
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents
    } = getInternalServices();
    const {
      agentId,
      agentParams
    } = request.body;
    const agentRegistry = agents.registry.asPublicRegistry();
    const agent = await agentRegistry.get({
      agentId,
      request
    });
    const agentResult = await agent.execute({
      agentParams: agentParams
    });
    return response.ok({
      body: agentResult
    });
  }));

  // stream agent events endpoint
  router.post({
    path: '/internal/onechat/agents/stream',
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        agentId: _configSchema.schema.string({}),
        agentParams: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
      })
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents
    } = getInternalServices();
    const {
      agentId,
      agentParams
    } = request.body;
    const agentRegistry = agents.registry.asPublicRegistry();
    const abortController = new AbortController();
    request.events.aborted$.subscribe(() => {
      abortController.abort();
    });
    const subject$ = new _rxjs.Subject();
    const agent = await agentRegistry.get({
      agentId,
      request
    });
    agent.execute({
      agentParams: agentParams,
      onEvent: event => {
        subject$.next(event);
      }
    }).then(result => {
      // TODO: should we emit an event with the final result?
      subject$.complete();
    }, error => {
      subject$.error(error);
    });
    return response.ok({
      body: (0, _sseUtilsServer.observableIntoEventSourceStream)(subject$, {
        signal: abortController.signal,
        logger
      })
    });
  }));
}