"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unknownToolProviderId = exports.toolDescriptorToIdentifier = exports.toStructuredToolIdentifier = exports.toSerializedToolIdentifier = exports.serializedPartsSeparator = exports.isStructuredToolIdentifier = exports.isSerializedToolIdentifier = exports.isPlainToolIdentifier = exports.createBuiltinToolId = exports.builtinToolProviderId = void 0;
var _errors = require("../base/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Represents a plain toolId, without source information attached to it.
 */

/**
 * Structured representation of a tool identifier.
 */

const serializedPartsSeparator = exports.serializedPartsSeparator = '::';

/**
 * The singleton sourceId used for all builtIn tools.
 */
const builtinToolProviderId = exports.builtinToolProviderId = 'builtIn';
/**
 * Unknown sourceId used from converting plain Ids to structured or serialized ids.
 */
const unknownToolProviderId = exports.unknownToolProviderId = 'unknown';

/**
 * Build a structured tool identifier for given builtin tool ID.
 */
const createBuiltinToolId = plainId => {
  return {
    toolId: plainId,
    providerId: builtinToolProviderId
  };
};

/**
 * String representation of {@link StructuredToolIdentifier}
 * Follow a `{toolId}||{sourceType}||{sourceId}` format.
 */

/**
 * Defines all possible shapes for a tool identifier.
 */
exports.createBuiltinToolId = createBuiltinToolId;
/**
 * Check if the given {@link ToolIdentifier} is a {@link SerializedToolIdentifier}
 */
const isSerializedToolIdentifier = identifier => {
  return typeof identifier === 'string' && identifier.split(serializedPartsSeparator).length === 2;
};

/**
 * Check if the given {@link ToolIdentifier} is a {@link StructuredToolIdentifier}
 */
exports.isSerializedToolIdentifier = isSerializedToolIdentifier;
const isStructuredToolIdentifier = identifier => {
  return typeof identifier === 'object' && 'toolId' in identifier && 'providerId' in identifier;
};

/**
 * Check if the given {@link ToolIdentifier} is a {@link PlainIdToolIdentifier}
 */
exports.isStructuredToolIdentifier = isStructuredToolIdentifier;
const isPlainToolIdentifier = identifier => {
  return typeof identifier === 'string' && identifier.split(serializedPartsSeparator).length === 1;
};

/**
 * Convert the given {@link ToolIdentifier} to a {@link SerializedToolIdentifier}
 */
exports.isPlainToolIdentifier = isPlainToolIdentifier;
const toSerializedToolIdentifier = identifier => {
  if (isSerializedToolIdentifier(identifier)) {
    return identifier;
  }
  if (isStructuredToolIdentifier(identifier)) {
    return `${identifier.providerId}::${identifier.toolId}`;
  }
  if (isPlainToolIdentifier(identifier)) {
    return `${unknownToolProviderId}::${identifier}`;
  }
  throw (0, _errors.createInternalError)(`Malformed tool identifier: "${identifier}"`);
};

/**
 * Convert the given {@link ToolIdentifier} to a {@link StructuredToolIdentifier}
 */
exports.toSerializedToolIdentifier = toSerializedToolIdentifier;
const toStructuredToolIdentifier = identifier => {
  if (isStructuredToolIdentifier(identifier)) {
    return identifier;
  }
  if (isSerializedToolIdentifier(identifier)) {
    const [providerId, toolId] = identifier.split(serializedPartsSeparator);
    return {
      toolId,
      providerId
    };
  }
  if (isPlainToolIdentifier(identifier)) {
    return {
      toolId: identifier,
      providerId: unknownToolProviderId
    };
  }
  throw (0, _errors.createInternalError)(`Malformed tool identifier: "${identifier}"`);
};

/**
 * Serializable representation of a tool, without its handler or schema.
 *
 * Use as a common base for browser-side and server-side tool types.
 */

/**
 * Metadata associated with a tool.
 *
 * Some of them are specified by the tool owner during registration,
 * others are automatically added by the framework.
 */
exports.toStructuredToolIdentifier = toStructuredToolIdentifier;
const toolDescriptorToIdentifier = tool => {
  return {
    toolId: tool.id,
    providerId: tool.meta.providerId
  };
};
exports.toolDescriptorToIdentifier = toolDescriptorToIdentifier;