"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getHoverItem = getHoverItem;
var _esqlAst = require("@kbn/esql-ast");
var _types = require("@kbn/esql-ast/src/types");
var _esqlValidationAutocomplete = require("@kbn/esql-validation-autocomplete");
var _autocomplete = require("@kbn/esql-validation-autocomplete/src/autocomplete/autocomplete");
var _util = require("@kbn/esql-validation-autocomplete/src/autocomplete/commands/enrich/util");
var _factories = require("@kbn/esql-validation-autocomplete/src/autocomplete/factories");
var _helper = require("@kbn/esql-validation-autocomplete/src/autocomplete/helper");
var _commands_helpers = require("@kbn/esql-validation-autocomplete/src/definitions/commands_helpers");
var _helpers = require("@kbn/esql-validation-autocomplete/src/shared/helpers");
var _resources_helpers = require("@kbn/esql-validation-autocomplete/src/shared/resources_helpers");
var _i18n = require("@kbn/i18n");
var _utils = require("../shared/utils");
var _helpers2 = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ACCEPTABLE_TYPES_HOVER = _i18n.i18n.translate('monaco.esql.hover.acceptableTypes', {
  defaultMessage: 'Acceptable types'
});
/**
 * @todo Monaco dependencies are not necesasry here: (1) replace {@link HoverMonacoModel}
 * by some generic `getText(): string` method; (2) replace {@link monaco.Position} by
 * `offset: number`.
 */
async function getHoverItem(model, position, callbacks) {
  var _callbacks$getVariabl;
  const fullText = model.getValue();
  const offset = (0, _utils.monacoPositionToOffset)(fullText, position);
  const {
    root
  } = (0, _esqlAst.parse)(fullText);
  let containingFunction;
  let node;
  _esqlAst.Walker.walk(root, {
    visitFunction: fn => {
      if ((0, _helpers.within)(offset, fn.location)) node = fn;
      if (fn.subtype === 'variadic-call') {
        const parentheses = {
          left: fullText.indexOf('(', fn.location.min),
          right: fn.location.max
        };
        if (parentheses.left < offset && parentheses.right > offset) containingFunction = fn;
      }
    },
    visitSource: (source, parent, walker) => {
      if ((0, _helpers.within)(offset, source.location)) {
        node = source;
        walker.abort();
      }
    },
    visitSingleAstItem: _node => {
      // ignore identifiers because we don't want to choose them as the node type
      // instead of the function node (functions can have an "operator" child which is
      // usually an identifer representing the name of the function)
      if (_node.type !== 'identifier' && (0, _helpers.within)(offset, _node.location)) {
        node = _node;
      }
    }
  });
  const hoverContent = {
    contents: []
  };
  if (!node) {
    return hoverContent;
  }
  const variables = callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getVariabl = callbacks.getVariables) === null || _callbacks$getVariabl === void 0 ? void 0 : _callbacks$getVariabl.call(callbacks);
  const variablesContent = (0, _helpers2.getVariablesHoverContent)(node, variables);
  if (variablesContent.length) {
    hoverContent.contents.push(...variablesContent);
  }
  if (containingFunction) {
    const argHints = await getHintForFunctionArg(containingFunction, root, fullText, offset, callbacks);
    hoverContent.contents.push(...argHints);
  }
  if (node.type === 'function') {
    const fnDefinition = (0, _esqlValidationAutocomplete.getFunctionDefinition)(node.name);
    if (fnDefinition) {
      hoverContent.contents.push(...[{
        value: (0, _esqlValidationAutocomplete.getFunctionSignatures)(fnDefinition)[0].declaration
      }, {
        value: fnDefinition.description
      }]);
    }
  }
  if (node.type === 'source' && node.sourceType === 'policy') {
    const source = node;
    const {
      getPolicyMetadata
    } = (0, _resources_helpers.getPolicyHelper)(callbacks);
    const policyMetadata = await getPolicyMetadata(node.name);
    if (policyMetadata) {
      hoverContent.contents.push(...[{
        value: `${_i18n.i18n.translate('monaco.esql.hover.policyIndexes', {
          defaultMessage: '**Indexes**'
        })}: ${policyMetadata.sourceIndices.join(', ')}`
      }, {
        value: `${_i18n.i18n.translate('monaco.esql.hover.policyMatchingField', {
          defaultMessage: '**Matching field**'
        })}: ${policyMetadata.matchField}`
      }, {
        value: `${_i18n.i18n.translate('monaco.esql.hover.policyEnrichedFields', {
          defaultMessage: '**Fields**'
        })}: ${policyMetadata.enrichFields.join(', ')}`
      }]);
    }
    if (!!source.prefix) {
      const mode = _commands_helpers.ENRICH_MODES.find(({
        name
      }) => '_' + name === source.prefix.valueUnquoted.toLowerCase());
      if (mode) {
        hoverContent.contents.push(...[{
          value: _util.modeDescription
        }, {
          value: `**${mode.name}**: ${mode.description}`
        }]);
      }
    }
  }
  return hoverContent;
}
async function getHintForFunctionArg(fnNode, root, query, offset, resourceRetriever) {
  const queryForFields = (0, _helper.getQueryForFields)(query, root);
  const {
    getFieldsMap
  } = (0, _autocomplete.getFieldsByTypeRetriever)(queryForFields, resourceRetriever);
  const fnDefinition = (0, _esqlValidationAutocomplete.getFunctionDefinition)(fnNode.name);
  // early exit on no hit
  if (!fnDefinition) {
    return [];
  }
  const fieldsMap = await getFieldsMap();
  const anyUserDefinedColumns = (0, _esqlValidationAutocomplete.collectUserDefinedColumns)(root.commands, fieldsMap, query);
  const references = {
    fields: fieldsMap,
    userDefinedColumns: anyUserDefinedColumns
  };
  const {
    typesToSuggestNext,
    enrichedArgs
  } = (0, _helper.getValidSignaturesAndTypesToSuggestNext)(fnNode, references, fnDefinition, query, offset);
  const hoveredArg = enrichedArgs[enrichedArgs.length - 1];
  const contents = [];
  if (hoveredArg && (0, _types.isESQLNamedParamLiteral)(hoveredArg)) {
    const bestMatch = _factories.TIME_SYSTEM_PARAMS.find(p => p.startsWith(hoveredArg.text));
    // We only know if it's start or end after first 3 characters (?t_s or ?t_e)
    if (hoveredArg.text.length > 3 && bestMatch) {
      Object.entries(_factories.TIME_SYSTEM_DESCRIPTIONS).forEach(([key, value]) => {
        contents.push({
          value: `**${key}**: ${value}`
        });
      });
    }
  }
  if (typesToSuggestNext.length > 0) {
    contents.push({
      value: `**${ACCEPTABLE_TYPES_HOVER}**: ${typesToSuggestNext.map(({
        type,
        constantOnly
      }) => `${constantOnly ? '_constant_ ' : ''}**${type}**` + (
      // If function arg is a constant date, helpfully suggest named time system params
      constantOnly && type === 'date' ? ` | ${_factories.TIME_SYSTEM_PARAMS.join(' | ')}` : '')).join(' | ')}`
    });
  }
  return contents;
}