"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useActionStatus = useActionStatus;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _hooks = require("../../../../hooks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useActionStatus(onAbortSuccess, refreshAgentActivity, nActions, dateFilter) {
  const [isFirstLoading, setIsFirstLoading] = (0, _react.useState)(true);
  const [currentActions, setCurrentActions] = (0, _react.useState)([]);
  const [actionCount, setActionCount] = (0, _react.useState)(0);
  const [areActionsFullyLoaded, setAreActionsFullyLoaded] = (0, _react.useState)(false);
  const {
    notifications,
    overlays
  } = (0, _hooks.useStartServices)();
  const loadActions = (0, _react.useCallback)(async () => {
    try {
      const res = await (0, _hooks.sendGetActionStatus)({
        perPage: nActions,
        date: dateFilter === null || dateFilter === void 0 ? void 0 : dateFilter.format()
      });
      setIsFirstLoading(false);
      if (res.error) {
        throw res.error;
      }
      if (!res.data) {
        throw new Error('No data');
      }
      setAreActionsFullyLoaded(actionCount < nActions);
      setActionCount(res.data.items.length);
      setCurrentActions(res.data.items);
    } catch (err) {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.fleet.actionStatus.fetchRequestError', {
          defaultMessage: 'An error happened while fetching action status'
        })
      });
    }
  }, [nActions, dateFilter, actionCount, notifications.toasts]);
  if (isFirstLoading) {
    loadActions();
  }
  (0, _react.useEffect)(() => {
    if (refreshAgentActivity) {
      loadActions();
    }
  }, [loadActions, refreshAgentActivity]);

  // second useEffect is needed for fetching actions when nAction changes
  (0, _react.useEffect)(() => {
    loadActions();
  }, [loadActions, nActions]);
  const abortUpgrade = (0, _react.useCallback)(async action => {
    try {
      const confirmRes = await overlays.openConfirm(_i18n.i18n.translate('xpack.fleet.currentUpgrade.confirmDescription', {
        defaultMessage: 'This action will cancel upgrade of {nbAgents, plural, one {# agent} other {# agents}}',
        values: {
          nbAgents: action.nbAgentsActioned - action.nbAgentsAck
        }
      }), {
        title: _i18n.i18n.translate('xpack.fleet.currentUpgrade.confirmTitle', {
          defaultMessage: 'Cancel upgrade?'
        })
      });
      if (!confirmRes) {
        return;
      }
      await (0, _hooks.sendPostCancelAction)(action.actionId);
      await Promise.all([loadActions(), onAbortSuccess()]);
    } catch (err) {
      notifications.toasts.addError(err, {
        title: _i18n.i18n.translate('xpack.fleet.currentUpgrade.abortRequestError', {
          defaultMessage: 'An error happened while cancelling upgrade'
        })
      });
    }
  }, [loadActions, notifications.toasts, overlays, onAbortSuccess]);
  return {
    currentActions,
    abortUpgrade,
    isFirstLoading,
    areActionsFullyLoaded
  };
}