"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectUserDefinedColumns = collectUserDefinedColumns;
exports.excludeUserDefinedColumnsFromCurrentCommand = excludeUserDefinedColumnsFromCurrentCommand;
var _visitor = require("@kbn/esql-ast/src/visitor");
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function addToUserDefinedColumnOccurrences(userDefinedColumns, instance) {
  if (!userDefinedColumns.has(instance.name)) {
    userDefinedColumns.set(instance.name, []);
  }
  const userDefinedColumnsOccurrencies = userDefinedColumns.get(instance.name);
  userDefinedColumnsOccurrencies.push(instance);
}
function addToUserDefinedColumns(oldArg, newArg, fields, userDefinedColumns) {
  if ((0, _helpers.isColumnItem)(oldArg) && (0, _helpers.isColumnItem)(newArg)) {
    const newUserDefinedColumn = {
      name: newArg.parts.join('.'),
      type: 'double' /* fallback to number */,
      location: newArg.location
    };
    // Now workout the exact type
    // it can be a rename of another userDefinedColumn as well
    const oldRef = fields.get(oldArg.parts.join('.')) || userDefinedColumns.get(oldArg.parts.join('.'));
    if (oldRef) {
      addToUserDefinedColumnOccurrences(userDefinedColumns, newUserDefinedColumn);
      newUserDefinedColumn.type = Array.isArray(oldRef) ? oldRef[0].type : oldRef.type;
    }
  }
}
function excludeUserDefinedColumnsFromCurrentCommand(commands, currentCommand, fieldsMap, queryString) {
  const anyUserDefinedColumns = collectUserDefinedColumns(commands, fieldsMap, queryString);
  const currentCommandUserDefinedColumns = collectUserDefinedColumns([currentCommand], fieldsMap, queryString);
  const resultUserDefinedColumns = new Map();
  anyUserDefinedColumns.forEach((value, key) => {
    if (!currentCommandUserDefinedColumns.has(key)) {
      resultUserDefinedColumns.set(key, value);
    }
  });
  return resultUserDefinedColumns;
}
function addUserDefinedColumnFromAssignment(assignOperation, userDefinedColumns, fields) {
  if ((0, _helpers.isColumnItem)(assignOperation.args[0])) {
    const rightHandSideArgType = (0, _helpers.getExpressionType)(assignOperation.args[1], fields, userDefinedColumns);
    addToUserDefinedColumnOccurrences(userDefinedColumns, {
      name: assignOperation.args[0].parts.join('.'),
      type: rightHandSideArgType /* fallback to number */,
      location: assignOperation.args[0].location
    });
  }
}
function addUserDefinedColumnFromExpression(expressionOperation, queryString, userDefinedColumns, fields) {
  if (!expressionOperation.text.includes(_constants.EDITOR_MARKER)) {
    const expressionText = queryString.substring(expressionOperation.location.min, expressionOperation.location.max + 1);
    const expressionType = (0, _helpers.getExpressionType)(expressionOperation, fields, userDefinedColumns);
    addToUserDefinedColumnOccurrences(userDefinedColumns, {
      name: expressionText,
      type: expressionType,
      location: expressionOperation.location
    });
  }
}
function collectUserDefinedColumns(ast, fields, queryString) {
  const userDefinedColumns = new Map();
  const visitor = new _visitor.Visitor().on('visitLiteralExpression', ctx => {
    // TODO - add these as userDefinedColumns
  }).on('visitExpression', _ctx => {}) // required for the types :shrug:
  .on('visitFunctionCallExpression', ctx => {
    const node = ctx.node;
    if (node.subtype === 'binary-expression' && node.name === 'where') {
      ctx.visitArgument(0, undefined);
      return;
    }
    if (node.name === 'as') {
      const [oldArg, newArg] = ctx.node.args;
      addToUserDefinedColumns(oldArg, newArg, fields, userDefinedColumns);
    } else if (node.name === '=') {
      addUserDefinedColumnFromAssignment(node, userDefinedColumns, fields);
    } else {
      addUserDefinedColumnFromExpression(node, queryString, userDefinedColumns, fields);
    }
  }).on('visitCommandOption', ctx => {
    if (ctx.node.name === 'by') {
      return [...ctx.visitArguments()];
    } else if (ctx.node.name === 'with') {
      for (const assignFn of ctx.node.args) {
        if ((0, _helpers.isFunctionItem)(assignFn)) {
          const [newArg, oldArg] = (assignFn === null || assignFn === void 0 ? void 0 : assignFn.args) || [];
          // TODO why is oldArg an array?
          if (Array.isArray(oldArg)) {
            addToUserDefinedColumns(oldArg[0], newArg, fields, userDefinedColumns);
          }
        }
      }
    }
  }).on('visitCommand', ctx => {
    const ret = [];
    if (['row', 'eval', 'stats', 'inlinestats', 'ts', 'rename'].includes(ctx.node.name)) {
      ret.push(...ctx.visitArgs());
    }
    if (['stats', 'inlinestats', 'enrich'].includes(ctx.node.name)) {
      // BY and WITH can contain userDefinedColumns
      ret.push(...ctx.visitOptions());
    }
    if (ctx.node.name === 'fork') {
      ret.push(...ctx.visitSubQueries());
    }
    return ret;
  }).on('visitQuery', ctx => [...ctx.visitCommands()]);
  visitor.visitQuery(ast);
  return userDefinedColumns;
}