"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateTabs = exports.updateTabAppStateAndGlobalState = exports.setTabs = exports.restoreTab = exports.initializeTabs = exports.disconnectTab = exports.clearAllTabs = void 0;
var _lodash = require("lodash");
var _selectors = require("../selectors");
var _internal_state = require("../internal_state");
var _runtime_state = require("../runtime_state");
var _constants = require("../../../../../../common/constants");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const setTabs = params => (dispatch, getState, {
  runtimeStateManager,
  tabsStorageManager,
  services: {
    profilesManager,
    ebtManager
  }
}) => {
  const previousState = getState();
  const previousTabs = (0, _selectors.selectAllTabs)(previousState);
  const removedTabs = (0, _lodash.differenceBy)(previousTabs, params.allTabs, differenceIterateeByTabId);
  const addedTabs = (0, _lodash.differenceBy)(params.allTabs, previousTabs, differenceIterateeByTabId);
  for (const tab of removedTabs) {
    dispatch(disconnectTab({
      tabId: tab.id
    }));
    delete runtimeStateManager.tabs.byId[tab.id];
  }
  for (const tab of addedTabs) {
    runtimeStateManager.tabs.byId[tab.id] = (0, _runtime_state.createTabRuntimeState)({
      profilesManager,
      ebtManager
    });
  }
  const selectedTabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, params.selectedTabId);
  if (selectedTabRuntimeState) {
    selectedTabRuntimeState.scopedEbtManager$.getValue().setAsActiveManager();
  }
  dispatch(_internal_state.internalStateSlice.actions.setTabs({
    ...params,
    recentlyClosedTabs: tabsStorageManager.getNRecentlyClosedTabs(
    // clean up the recently closed tabs if the same ids are present in next open tabs
    (0, _lodash.differenceBy)(params.recentlyClosedTabs, params.allTabs, differenceIterateeByTabId), removedTabs)
  }));
};
exports.setTabs = setTabs;
const updateTabs = ({
  items,
  selectedItem
}) => async (dispatch, getState, {
  services,
  runtimeStateManager,
  urlStateStorage
}) => {
  var _selectedItem$id;
  const currentState = getState();
  const currentTab = (0, _selectors.selectTab)(currentState, currentState.tabs.unsafeCurrentId);
  const updatedTabs = items.map(item => {
    const existingTab = (0, _selectors.selectTab)(currentState, item.id);
    const tab = {
      ..._internal_state.defaultTabState,
      ...existingTab,
      ...(0, _lodash.pick)(item, 'id', 'label')
    };
    if (item.duplicatedFromId) {
      var _selectTabRuntimeAppS, _selectTabRuntimeGlob;
      const existingTabToDuplicate = (0, _selectors.selectTab)(currentState, item.duplicatedFromId);
      tab.initialAppState = (_selectTabRuntimeAppS = (0, _runtime_state.selectTabRuntimeAppState)(runtimeStateManager, item.duplicatedFromId)) !== null && _selectTabRuntimeAppS !== void 0 ? _selectTabRuntimeAppS : (0, _lodash.cloneDeep)(existingTabToDuplicate.initialAppState);
      tab.initialGlobalState = (_selectTabRuntimeGlob = (0, _runtime_state.selectTabRuntimeGlobalState)(runtimeStateManager, item.duplicatedFromId)) !== null && _selectTabRuntimeGlob !== void 0 ? _selectTabRuntimeGlob : (0, _lodash.cloneDeep)(existingTabToDuplicate.initialGlobalState);
    }
    return tab;
  });
  if ((selectedItem === null || selectedItem === void 0 ? void 0 : selectedItem.id) !== currentTab.id) {
    const previousTabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, currentTab.id);
    const previousTabStateContainer = previousTabRuntimeState.stateContainer$.getValue();
    previousTabStateContainer === null || previousTabStateContainer === void 0 ? void 0 : previousTabStateContainer.actions.stopSyncing();
    const nextTab = selectedItem ? (0, _selectors.selectTab)(currentState, selectedItem.id) : undefined;
    const nextTabRuntimeState = selectedItem ? (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, selectedItem.id) : undefined;
    const nextTabStateContainer = nextTabRuntimeState === null || nextTabRuntimeState === void 0 ? void 0 : nextTabRuntimeState.stateContainer$.getValue();
    if (nextTab && nextTabStateContainer) {
      const {
        timeRange,
        refreshInterval,
        filters: globalFilters
      } = nextTab.lastPersistedGlobalState;
      const appState = nextTabStateContainer.appState.getState();
      const {
        filters: appFilters,
        query
      } = appState;
      await urlStateStorage.set(_constants.GLOBAL_STATE_URL_KEY, {
        time: timeRange,
        refreshInterval,
        filters: globalFilters
      });
      await urlStateStorage.set(_constants.APP_STATE_URL_KEY, appState);
      services.timefilter.setTime(timeRange !== null && timeRange !== void 0 ? timeRange : services.timefilter.getTimeDefaults());
      services.timefilter.setRefreshInterval(refreshInterval !== null && refreshInterval !== void 0 ? refreshInterval : services.timefilter.getRefreshIntervalDefaults());
      services.filterManager.setGlobalFilters((0, _lodash.cloneDeep)(globalFilters !== null && globalFilters !== void 0 ? globalFilters : []));
      services.filterManager.setAppFilters((0, _lodash.cloneDeep)(appFilters !== null && appFilters !== void 0 ? appFilters : []));
      services.data.query.queryString.setQuery(query !== null && query !== void 0 ? query : services.data.query.queryString.getDefaultQuery());
      nextTabStateContainer.actions.initializeAndSync();
    } else {
      await urlStateStorage.set(_constants.GLOBAL_STATE_URL_KEY, null);
      await urlStateStorage.set(_constants.APP_STATE_URL_KEY, null);
    }
  }
  dispatch(setTabs({
    allTabs: updatedTabs,
    selectedTabId: (_selectedItem$id = selectedItem === null || selectedItem === void 0 ? void 0 : selectedItem.id) !== null && _selectedItem$id !== void 0 ? _selectedItem$id : currentTab.id,
    recentlyClosedTabs: (0, _selectors.selectRecentlyClosedTabs)(currentState)
  }));
};
exports.updateTabs = updateTabs;
const updateTabAppStateAndGlobalState = ({
  tabId
}) => (dispatch, _, {
  runtimeStateManager
}) => {
  dispatch(_internal_state.internalStateSlice.actions.setTabAppStateAndGlobalState({
    tabId,
    appState: (0, _runtime_state.selectTabRuntimeAppState)(runtimeStateManager, tabId),
    globalState: (0, _runtime_state.selectTabRuntimeGlobalState)(runtimeStateManager, tabId)
  }));
};
exports.updateTabAppStateAndGlobalState = updateTabAppStateAndGlobalState;
const initializeTabs = ({
  userId,
  spaceId
}) => (dispatch, _, {
  tabsStorageManager
}) => {
  const initialTabsState = tabsStorageManager.loadLocally({
    userId,
    spaceId,
    defaultTabState: _internal_state.defaultTabState
  });
  dispatch(setTabs(initialTabsState));
};
exports.initializeTabs = initializeTabs;
const clearAllTabs = () => dispatch => {
  const defaultTab = {
    ..._internal_state.defaultTabState,
    ...(0, _utils.createTabItem)([])
  };
  return dispatch(updateTabs({
    items: [defaultTab],
    selectedItem: defaultTab
  }));
};
exports.clearAllTabs = clearAllTabs;
const restoreTab = ({
  restoreTabId
}) => (dispatch, getState) => {
  const currentState = getState();
  if (restoreTabId === currentState.tabs.unsafeCurrentId) {
    return;
  }
  const currentTabs = (0, _selectors.selectAllTabs)(currentState);
  const currentTab = (0, _selectors.selectTab)(currentState, currentState.tabs.unsafeCurrentId);
  let items = currentTabs;
  // search among open tabs
  let selectedItem = items.find(tab => tab.id === restoreTabId);
  if (!selectedItem) {
    // search among recently closed tabs
    const recentlyClosedTabs = (0, _selectors.selectRecentlyClosedTabs)(currentState);
    const closedTab = recentlyClosedTabs.find(tab => tab.id === restoreTabId);
    if (closedTab) {
      // reopening one of the closed tabs
      selectedItem = (0, _lodash.omit)(closedTab, 'closedAt');
      items = [...items, closedTab];
    }
  }
  return dispatch(updateTabs({
    items,
    selectedItem: selectedItem || currentTab
  }));
};
exports.restoreTab = restoreTab;
const disconnectTab = ({
  tabId
}) => (_, __, {
  runtimeStateManager
}) => {
  var _tabRuntimeState$cust;
  const tabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, tabId);
  const stateContainer = tabRuntimeState.stateContainer$.getValue();
  stateContainer === null || stateContainer === void 0 ? void 0 : stateContainer.dataState.cancel();
  stateContainer === null || stateContainer === void 0 ? void 0 : stateContainer.actions.stopSyncing();
  (_tabRuntimeState$cust = tabRuntimeState.customizationService$.getValue()) === null || _tabRuntimeState$cust === void 0 ? void 0 : _tabRuntimeState$cust.cleanup();
};
exports.disconnectTab = disconnectTab;
function differenceIterateeByTabId(tab) {
  return tab.id;
}