"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FavoritesService = exports.FavoritesLimitExceededError = void 0;
var _contentManagementFavoritesCommon = require("@kbn/content-management-favorites-common");
var _server = require("@kbn/core/server");
var _favorites_saved_object = require("./favorites_saved_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// eslint-disable-next-line max-classes-per-file

class FavoritesService {
  constructor(type, userId, deps) {
    this.type = type;
    this.userId = userId;
    this.deps = deps;
    if (!this.userId || !this.type) {
      // This should never happen, but just in case let's do a runtime check
      throw new Error('userId and object type are required to use a favorite service');
    }
    if (!this.deps.favoritesRegistry.hasType(this.type)) {
      throw new Error(`Favorite type ${this.type} is not registered`);
    }
  }
  async getFavorites() {
    var _favoritesSavedObject, _favoritesSavedObject2, _favoritesSavedObject3;
    const favoritesSavedObject = await this.getFavoritesSavedObject();
    const favoriteIds = (_favoritesSavedObject = favoritesSavedObject === null || favoritesSavedObject === void 0 ? void 0 : (_favoritesSavedObject2 = favoritesSavedObject.attributes) === null || _favoritesSavedObject2 === void 0 ? void 0 : _favoritesSavedObject2.favoriteIds) !== null && _favoritesSavedObject !== void 0 ? _favoritesSavedObject : [];
    const favoriteMetadata = favoritesSavedObject === null || favoritesSavedObject === void 0 ? void 0 : (_favoritesSavedObject3 = favoritesSavedObject.attributes) === null || _favoritesSavedObject3 === void 0 ? void 0 : _favoritesSavedObject3.favoriteMetadata;
    return {
      favoriteIds,
      favoriteMetadata
    };
  }

  /**
   * @throws {FavoritesLimitExceededError}
   */
  async addFavorite({
    id,
    metadata
  }) {
    let favoritesSavedObject = await this.getFavoritesSavedObject();
    if (!favoritesSavedObject) {
      favoritesSavedObject = await this.deps.savedObjectClient.create(_favorites_saved_object.favoritesSavedObjectType.name, {
        userId: this.userId,
        type: this.type,
        favoriteIds: [id],
        ...(metadata ? {
          favoriteMetadata: {
            [id]: metadata
          }
        } : {})
      }, {
        id: this.getFavoriteSavedObjectId()
      });
      return {
        favoriteIds: favoritesSavedObject.attributes.favoriteIds,
        favoriteMetadata: favoritesSavedObject.attributes.favoriteMetadata
      };
    } else {
      var _favoritesSavedObject4, _favoritesSavedObject5;
      if (((_favoritesSavedObject4 = favoritesSavedObject.attributes.favoriteIds) !== null && _favoritesSavedObject4 !== void 0 ? _favoritesSavedObject4 : []).length >= _contentManagementFavoritesCommon.FAVORITES_LIMIT) {
        throw new FavoritesLimitExceededError();
      }
      const newFavoriteIds = [...((_favoritesSavedObject5 = favoritesSavedObject.attributes.favoriteIds) !== null && _favoritesSavedObject5 !== void 0 ? _favoritesSavedObject5 : []).filter(favoriteId => favoriteId !== id), id];
      const newFavoriteMetadata = metadata ? {
        ...favoritesSavedObject.attributes.favoriteMetadata,
        [id]: metadata
      } : undefined;
      await this.deps.savedObjectClient.update(_favorites_saved_object.favoritesSavedObjectType.name, favoritesSavedObject.id, {
        favoriteIds: newFavoriteIds,
        ...(newFavoriteMetadata ? {
          favoriteMetadata: newFavoriteMetadata
        } : {})
      }, {
        version: favoritesSavedObject.version
      });
      return {
        favoriteIds: newFavoriteIds,
        favoriteMetadata: newFavoriteMetadata
      };
    }
  }
  async removeFavorite({
    id
  }) {
    var _favoritesSavedObject6;
    const favoritesSavedObject = await this.getFavoritesSavedObject();
    if (!favoritesSavedObject) {
      return {
        favoriteIds: []
      };
    }
    const newFavoriteIds = ((_favoritesSavedObject6 = favoritesSavedObject.attributes.favoriteIds) !== null && _favoritesSavedObject6 !== void 0 ? _favoritesSavedObject6 : []).filter(favoriteId => favoriteId !== id);
    const newFavoriteMetadata = favoritesSavedObject.attributes.favoriteMetadata ? {
      ...favoritesSavedObject.attributes.favoriteMetadata
    } : undefined;
    if (newFavoriteMetadata) {
      delete newFavoriteMetadata[id];
    }
    await this.deps.savedObjectClient.update(_favorites_saved_object.favoritesSavedObjectType.name, favoritesSavedObject.id, {
      ...favoritesSavedObject.attributes,
      favoriteIds: newFavoriteIds,
      ...(newFavoriteMetadata ? {
        favoriteMetadata: newFavoriteMetadata
      } : {})
    }, {
      version: favoritesSavedObject.version,
      // We don't want to merge the attributes here because we want to remove the keys from the metadata
      mergeAttributes: false
    });
    return {
      favoriteIds: newFavoriteIds,
      favoriteMetadata: newFavoriteMetadata
    };
  }
  async getFavoritesSavedObject() {
    try {
      const favoritesSavedObject = await this.deps.savedObjectClient.get(_favorites_saved_object.favoritesSavedObjectType.name, this.getFavoriteSavedObjectId());
      return favoritesSavedObject;
    } catch (e) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        return null;
      }
      throw e;
    }
  }
  getFavoriteSavedObjectId() {
    return `${this.type}:${this.userId}`;
  }
}
exports.FavoritesService = FavoritesService;
class FavoritesLimitExceededError extends Error {
  constructor() {
    super(`Limit reached: This list can contain a maximum of ${_contentManagementFavoritesCommon.FAVORITES_LIMIT} items. Please remove an item before adding a new one.`);
    this.name = 'FavoritesLimitExceededError';
    Object.setPrototypeOf(this, FavoritesLimitExceededError.prototype); // For TypeScript compatibility
  }
}
exports.FavoritesLimitExceededError = FavoritesLimitExceededError;