"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGraph = void 0;
var _lodash = require("lodash");
var _uuid = require("uuid");
var _latest = require("@kbn/cloud-security-posture-common/types/graph/latest");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getGraph = async ({
  services: {
    esClient,
    logger
  },
  query: {
    originEventIds,
    spaceId = 'default',
    start,
    end,
    esQuery
  },
  showUnknownTarget,
  nodesLimit
}) => {
  logger.trace(`Fetching graph for [originEventIds: ${originEventIds.join(', ')}] in [spaceId: ${spaceId}]`);
  const results = await fetchGraph({
    esClient,
    showUnknownTarget,
    logger,
    start,
    end,
    originEventIds,
    esQuery
  });

  // Convert results into set of nodes and edges
  return parseRecords(logger, results.records, nodesLimit);
};
exports.getGraph = getGraph;
const parseRecords = (logger, records, nodesLimit) => {
  const ctx = {
    nodesLimit,
    logger,
    nodesMap: {},
    edgeLabelsNodes: {},
    edgesMap: {},
    labelEdges: {},
    messages: []
  };
  logger.trace(`Parsing records [length: ${records.length}] [nodesLimit: ${nodesLimit !== null && nodesLimit !== void 0 ? nodesLimit : 'none'}]`);
  createNodes(records, ctx);
  createEdgesAndGroups(ctx);
  logger.trace(`Parsed [nodes: ${Object.keys(ctx.nodesMap).length}, edges: ${Object.keys(ctx.edgesMap).length}]`);

  // Sort groups to be first (fixes minor layout issue)
  const nodes = sortNodes(ctx.nodesMap);
  return {
    nodes,
    edges: Object.values(ctx.edgesMap),
    messages: ctx.messages.length > 0 ? ctx.messages : undefined
  };
};
const fetchGraph = async ({
  esClient,
  logger,
  start,
  end,
  originEventIds,
  showUnknownTarget,
  esQuery
}) => {
  const originAlertIds = originEventIds.filter(originEventId => originEventId.isAlert);
  const query = `FROM logs-*
| WHERE event.action IS NOT NULL AND actor.entity.id IS NOT NULL
| EVAL isOrigin = ${originEventIds.length > 0 ? `event.id in (${originEventIds.map((_id, idx) => `?og_id${idx}`).join(', ')})` : 'false'}
| EVAL isOriginAlert = isOrigin AND ${originAlertIds.length > 0 ? `event.id in (${originAlertIds.map((_id, idx) => `?og_alrt_id${idx}`).join(', ')})` : 'false'}
| STATS badge = COUNT(*),
  ips = VALUES(related.ip),
  // hosts = VALUES(related.hosts),
  users = VALUES(related.user)
    BY actorIds = actor.entity.id,
      action = event.action,
      targetIds = target.entity.id,
      isOrigin,
      isOriginAlert
| LIMIT 1000
| SORT isOrigin DESC`;
  logger.trace(`Executing query [${query}]`);
  const eventIds = originEventIds.map(originEventId => originEventId.id);
  return await esClient.asCurrentUser.helpers.esql({
    columnar: false,
    filter: buildDslFilter(eventIds, showUnknownTarget, start, end, esQuery),
    query,
    // @ts-ignore - types are not up to date
    params: [...originEventIds.map((originEventId, idx) => ({
      [`og_id${idx}`]: originEventId.id
    })), ...originEventIds.filter(originEventId => originEventId.isAlert).map((originEventId, idx) => ({
      [`og_alrt_id${idx}`]: originEventId.id
    }))]
  }).toRecords();
};
const buildDslFilter = (eventIds, showUnknownTarget, start, end, esQuery) => {
  var _esQuery$bool$filter, _esQuery$bool$must, _esQuery$bool$should, _esQuery$bool$must_no;
  return {
    bool: {
      filter: [{
        range: {
          '@timestamp': {
            gte: start,
            lte: end
          }
        }
      }, ...(showUnknownTarget ? [] : [{
        exists: {
          field: 'target.entity.id'
        }
      }]), {
        bool: {
          should: [...(esQuery !== null && esQuery !== void 0 && (_esQuery$bool$filter = esQuery.bool.filter) !== null && _esQuery$bool$filter !== void 0 && _esQuery$bool$filter.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$must = esQuery.bool.must) !== null && _esQuery$bool$must !== void 0 && _esQuery$bool$must.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$should = esQuery.bool.should) !== null && _esQuery$bool$should !== void 0 && _esQuery$bool$should.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$must_no = esQuery.bool.must_not) !== null && _esQuery$bool$must_no !== void 0 && _esQuery$bool$must_no.length ? [esQuery] : []), {
            terms: {
              'event.id': eventIds
            }
          }],
          minimum_should_match: 1
        }
      }]
    }
  };
};
const createNodes = (records, context) => {
  const {
    nodesMap,
    edgeLabelsNodes,
    labelEdges
  } = context;
  for (const record of records) {
    if (context.nodesLimit !== undefined && Object.keys(nodesMap).length >= context.nodesLimit) {
      context.logger.debug(`Reached nodes limit [limit: ${context.nodesLimit}] [current: ${Object.keys(nodesMap).length}]`);
      context.messages.push(_latest.ApiMessageCode.ReachedNodesLimit);
      break;
    }
    const {
      ips,
      hosts,
      users,
      actorIds,
      action,
      targetIds,
      isOriginAlert
    } = record;
    const actorIdsArray = (0, _lodash.castArray)(actorIds);
    const targetIdsArray = (0, _lodash.castArray)(targetIds);
    const unknownTargets = [];

    // Ensure all targets has an id (target can return null from the query)
    targetIdsArray.forEach((id, idx) => {
      if (!id) {
        targetIdsArray[idx] = `unknown ${(0, _uuid.v4)()}`;
        unknownTargets.push(targetIdsArray[idx]);
      }
    });

    // Create entity nodes
    [...actorIdsArray, ...targetIdsArray].forEach(id => {
      if (nodesMap[id] === undefined) {
        nodesMap[id] = {
          id,
          label: unknownTargets.includes(id) ? 'Unknown' : undefined,
          color: 'primary',
          ...determineEntityNodeShape(id, (0, _lodash.castArray)(ips !== null && ips !== void 0 ? ips : []), (0, _lodash.castArray)(hosts !== null && hosts !== void 0 ? hosts : []), (0, _lodash.castArray)(users !== null && users !== void 0 ? users : []))
        };
      }
    });

    // Create label nodes
    for (const actorId of actorIdsArray) {
      for (const targetId of targetIdsArray) {
        const edgeId = `a(${actorId})-b(${targetId})`;
        if (edgeLabelsNodes[edgeId] === undefined) {
          edgeLabelsNodes[edgeId] = [];
        }
        const labelNode = {
          id: edgeId + `label(${action})`,
          label: action,
          color: isOriginAlert ? 'danger' : 'primary',
          shape: 'label'
        };
        nodesMap[labelNode.id] = labelNode;
        edgeLabelsNodes[edgeId].push(labelNode.id);
        labelEdges[labelNode.id] = {
          source: actorId,
          target: targetId,
          edgeType: 'solid'
        };
      }
    }
  }
};
const determineEntityNodeShape = (actorId, ips, hosts, users) => {
  // If actor is a user return ellipse
  if (users.includes(actorId)) {
    return {
      shape: 'ellipse',
      icon: 'user'
    };
  }

  // If actor is a host return hexagon
  if (hosts.includes(actorId)) {
    return {
      shape: 'hexagon',
      icon: 'storage'
    };
  }

  // If actor is an IP return diamond
  if (ips.includes(actorId)) {
    return {
      shape: 'diamond',
      icon: 'globe'
    };
  }
  return {
    shape: 'hexagon'
  };
};
const sortNodes = nodesMap => {
  const groupNodes = [];
  const otherNodes = [];
  for (const node of Object.values(nodesMap)) {
    if (node.shape === 'group') {
      groupNodes.push(node);
    } else {
      otherNodes.push(node);
    }
  }
  return [...groupNodes, ...otherNodes];
};
const createEdgesAndGroups = context => {
  const {
    edgeLabelsNodes,
    edgesMap,
    nodesMap,
    labelEdges
  } = context;
  Object.entries(edgeLabelsNodes).forEach(([edgeId, edgeLabelsIds]) => {
    // When there's more than one edge label, create a group node
    if (edgeLabelsIds.length === 1) {
      const edgeLabelId = edgeLabelsIds[0];
      connectEntitiesAndLabelNode(edgesMap, nodesMap, labelEdges[edgeLabelId].source, edgeLabelId, labelEdges[edgeLabelId].target, labelEdges[edgeLabelId].edgeType);
    } else {
      const groupNode = {
        id: `grp(${edgeId})`,
        shape: 'group'
      };
      nodesMap[groupNode.id] = groupNode;
      let groupEdgesColor = 'subdued';

      // Order of creation matters when using dagre layout, first create edges to the group node,
      // then connect the group node to the label nodes
      connectEntitiesAndLabelNode(edgesMap, nodesMap, labelEdges[edgeLabelsIds[0]].source, groupNode.id, labelEdges[edgeLabelsIds[0]].target, 'solid', groupEdgesColor);
      edgeLabelsIds.forEach(edgeLabelId => {
        nodesMap[edgeLabelId].parentId = groupNode.id;
        connectEntitiesAndLabelNode(edgesMap, nodesMap, groupNode.id, edgeLabelId, groupNode.id, labelEdges[edgeLabelId].edgeType);
        if (nodesMap[edgeLabelId].color === 'danger') {
          groupEdgesColor = 'danger';
        }
      });
    }
  });
};
const connectEntitiesAndLabelNode = (edgesMap, nodesMap, sourceNodeId, labelNodeId, targetNodeId, edgeType = 'solid', colorOverride) => {
  [connectNodes(nodesMap, sourceNodeId, labelNodeId, edgeType, colorOverride), connectNodes(nodesMap, labelNodeId, targetNodeId, edgeType, colorOverride)].forEach(edge => {
    edgesMap[edge.id] = edge;
  });
};
const connectNodes = (nodesMap, sourceNodeId, targetNodeId, edgeType = 'solid', colorOverride) => {
  const sourceNode = nodesMap[sourceNodeId];
  const targetNode = nodesMap[targetNodeId];
  const color = sourceNode.shape === 'label' && sourceNode.color === 'danger' || targetNode.shape === 'label' && targetNode.color === 'danger' ? 'danger' : 'subdued';
  return {
    id: `a(${sourceNodeId})-b(${targetNodeId})`,
    source: sourceNodeId,
    target: targetNodeId,
    color: colorOverride !== null && colorOverride !== void 0 ? colorOverride : color,
    type: edgeType
  };
};