/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.indexing;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.indexing.IndexerJobStats;
import org.elasticsearch.xpack.core.indexing.IndexerState;
import org.elasticsearch.xpack.core.indexing.IterationResult;

public abstract class AsyncTwoPhaseIndexer<JobPosition, JobStats extends IndexerJobStats> {
    private static final Logger logger = LogManager.getLogger((String)AsyncTwoPhaseIndexer.class.getName());
    private static final TimeValue MAX_THROTTLE_WAIT_TIME = TimeValue.timeValueHours(1L);
    private static final TimeValue MIN_THROTTLE_WAIT_TIME = TimeValue.timeValueMillis(10L);
    private final ActionListener<SearchResponse> searchResponseListener = ActionListener.wrap(this::onSearchResponse, this::finishWithSearchFailure);
    private final JobStats stats;
    private final AtomicReference<IndexerState> state;
    private final AtomicReference<JobPosition> position;
    private final ThreadPool threadPool;
    private final Object lock;
    private final AtomicBoolean isJobFinishing;
    private volatile float currentMaxDocsPerSecond;
    private volatile long lastSearchStartTimeNanos = 0L;
    private volatile long lastDocCount = 0L;
    private volatile ScheduledRunnable scheduledNextSearch;

    protected AsyncTwoPhaseIndexer(ThreadPool threadPool, AtomicReference<IndexerState> initialState, JobPosition initialPosition, JobStats jobStats) {
        this(threadPool, initialState, initialPosition, jobStats, new Object());
    }

    protected AsyncTwoPhaseIndexer(ThreadPool threadPool, AtomicReference<IndexerState> initialState, JobPosition initialPosition, JobStats jobStats, Object lock) {
        this.threadPool = threadPool;
        this.state = initialState;
        this.position = new AtomicReference<JobPosition>(initialPosition);
        this.stats = jobStats;
        this.lock = lock;
        this.isJobFinishing = new AtomicBoolean(false);
    }

    public IndexerState getState() {
        return this.state.get();
    }

    public JobPosition getPosition() {
        return this.position.get();
    }

    public JobStats getStats() {
        return this.stats;
    }

    public IndexerState start() {
        if (this.state.compareAndSet(IndexerState.STOPPED, IndexerState.STARTED)) {
            this.isJobFinishing.set(false);
        }
        return this.state.get();
    }

    public IndexerState stop() {
        IndexerState indexerState = this.state.updateAndGet(previousState -> {
            if (previousState == IndexerState.INDEXING) {
                return IndexerState.STOPPING;
            }
            if (previousState == IndexerState.STARTED) {
                return IndexerState.STOPPED;
            }
            return previousState;
        });
        this.runSearchImmediately();
        return indexerState;
    }

    public boolean abort() {
        IndexerState prevState = this.state.getAndUpdate(prev -> IndexerState.ABORTING);
        return prevState == IndexerState.STOPPED || prevState == IndexerState.STARTED;
    }

    public boolean maybeTriggerAsyncJob(long now) {
        Object object = this.lock;
        synchronized (object) {
            IndexerState currentState = this.state.get();
            switch (currentState) {
                case INDEXING: 
                case STOPPING: 
                case ABORTING: {
                    logger.warn("Schedule was triggered for job [" + this.getJobId() + "], but prior indexer is still running (with state [" + String.valueOf(currentState) + "]");
                    return false;
                }
                case STOPPED: {
                    logger.debug("Schedule was triggered for job [" + this.getJobId() + "] but job is stopped.  Ignoring trigger.");
                    return false;
                }
                case STARTED: {
                    logger.debug("Schedule was triggered for job [" + this.getJobId() + "], state: [" + String.valueOf(currentState) + "]");
                    ((IndexerJobStats)this.stats).incrementNumInvocations(1L);
                    if (this.startJob()) {
                        this.threadPool.executor("generic").execute(() -> this.onStart(now, ActionListener.wrap(r -> {
                            assert (r != null);
                            if (r.booleanValue()) {
                                this.nextSearch();
                            } else {
                                this.onFinish(this.finishJobListener());
                            }
                        }, this::finishWithFailure)));
                        logger.debug("Beginning to index [" + this.getJobId() + "], state: [" + String.valueOf(currentState) + "]");
                        return true;
                    }
                    return false;
                }
            }
            logger.warn("Encountered unexpected state [" + String.valueOf(currentState) + "] while indexing");
            throw new IllegalStateException("Job encountered an illegal state [" + String.valueOf(currentState) + "]");
        }
    }

    private boolean startJob() {
        if (!this.isJobFinishing.get() && this.state.compareAndSet(IndexerState.STARTED, IndexerState.INDEXING)) {
            return true;
        }
        Supplier[] supplierArray = new Supplier[2];
        supplierArray[0] = this.state::get;
        supplierArray[1] = this.isJobFinishing::get;
        logger.debug("Could not start job because current state is [{}] and another job may be finishing [{}]", supplierArray);
        return false;
    }

    private void finishJob() {
        this.isJobFinishing.set(true);
        this.doSaveState(this.finishAndSetState(), this.position.get(), () -> {
            this.afterFinishOrFailure();
            this.isJobFinishing.set(false);
        });
    }

    private <T> ActionListener<T> finishJobListener() {
        return ActionListener.wrap(r -> this.finishJob(), e -> this.finishJob());
    }

    protected boolean triggerSaveState() {
        return ((IndexerJobStats)this.stats).getNumPages() > 0L && ((IndexerJobStats)this.stats).getNumPages() % 50L == 0L;
    }

    protected void rethrottle() {
        if (this.getMaxDocsPerSecond() == this.currentMaxDocsPerSecond) {
            return;
        }
        this.reQueueThrottledSearch();
    }

    protected void runSearchImmediately() {
        ScheduledRunnable runnable = this.scheduledNextSearch;
        if (runnable != null) {
            runnable.reschedule(TimeValue.ZERO);
        }
    }

    protected long getTimeNanos() {
        return System.nanoTime();
    }

    protected ScheduledRunnable getScheduledNextSearch() {
        return this.scheduledNextSearch;
    }

    protected float getMaxDocsPerSecond() {
        return -1.0f;
    }

    protected abstract String getJobId();

    protected abstract IterationResult<JobPosition> doProcess(SearchResponse var1);

    protected abstract void onStart(long var1, ActionListener<Boolean> var3);

    protected abstract void doNextSearch(long var1, ActionListener<SearchResponse> var3);

    protected abstract void doNextBulk(BulkRequest var1, ActionListener<BulkResponse> var2);

    protected abstract void doSaveState(IndexerState var1, JobPosition var2, Runnable var3);

    protected abstract void onFailure(Exception var1);

    protected abstract void onFinish(ActionListener<Void> var1);

    protected void afterFinishOrFailure() {
    }

    protected void onStop() {
    }

    protected abstract void onAbort();

    private void finishWithSearchFailure(Exception exc) {
        ((IndexerJobStats)this.stats).incrementSearchFailures();
        this.finishWithFailure(exc);
    }

    private void finishWithIndexingFailure(Exception exc) {
        ((IndexerJobStats)this.stats).incrementIndexingFailures();
        this.finishWithFailure(exc);
    }

    private void finishWithFailure(Exception exc) {
        this.onFailure(exc);
        this.finishJob();
    }

    private IndexerState finishAndSetState() {
        AtomicBoolean callOnStop = new AtomicBoolean(false);
        AtomicBoolean callOnAbort = new AtomicBoolean(false);
        IndexerState updatedState = this.state.updateAndGet(prev -> {
            callOnAbort.set(false);
            callOnStop.set(false);
            switch (prev) {
                case INDEXING: {
                    return IndexerState.STARTED;
                }
                case STOPPING: {
                    callOnStop.set(true);
                    return IndexerState.STOPPED;
                }
                case ABORTING: {
                    callOnAbort.set(true);
                    return IndexerState.ABORTING;
                }
                case STOPPED: {
                    return IndexerState.STOPPED;
                }
            }
            throw new IllegalStateException("Indexer job encountered an illegal state [" + String.valueOf(prev) + "]");
        });
        if (callOnStop.get()) {
            this.onStop();
        } else if (callOnAbort.get()) {
            this.onAbort();
        }
        return updatedState;
    }

    private void onSearchResponse(SearchResponse searchResponse) {
        block10: {
            ((IndexerJobStats)this.stats).markEndSearch();
            try {
                if (!this.checkState(this.getState())) {
                    return;
                }
                if (searchResponse == null) {
                    logger.debug("No indexing necessary for job [{}], saving state and shutting down.", (Object)this.getJobId());
                    this.onFinish(this.finishJobListener());
                    return;
                }
                assert (searchResponse.getShardFailures().length == 0);
                ((IndexerJobStats)this.stats).markStartProcessing();
                ((IndexerJobStats)this.stats).incrementNumPages(1L);
                long numDocumentsBefore = ((IndexerJobStats)this.stats).getNumDocuments();
                IterationResult<JobPosition> iterationResult = this.doProcess(searchResponse);
                this.lastDocCount = ((IndexerJobStats)this.stats).getNumDocuments() - numDocumentsBefore;
                if (iterationResult.isDone()) {
                    logger.debug("Finished indexing for job [{}], saving state and shutting down.", (Object)this.getJobId());
                    this.position.set(iterationResult.getPosition());
                    ((IndexerJobStats)this.stats).markEndProcessing();
                    this.onFinish(this.finishJobListener());
                    return;
                }
                BulkRequest bulkRequest = new BulkRequest();
                iterationResult.getToIndex().forEach(bulkRequest::add);
                ((IndexerJobStats)this.stats).markEndProcessing();
                if (bulkRequest.numberOfActions() > 0) {
                    ((IndexerJobStats)this.stats).markStartIndexing();
                    this.doNextBulk(bulkRequest, ActionListener.wrap(bulkResponse -> {
                        if (bulkResponse.hasFailures()) {
                            logger.warn("Error while attempting to bulk index documents: {}", (Object)bulkResponse.buildFailureMessage());
                        }
                        ((IndexerJobStats)this.stats).incrementNumOutputDocuments(bulkResponse.getItems().length);
                        Object newPosition = iterationResult.getPosition();
                        this.position.set(newPosition);
                        this.onBulkResponse((BulkResponse)bulkResponse, newPosition);
                    }, this::finishWithIndexingFailure));
                    break block10;
                }
                try {
                    JobPosition newPosition = iterationResult.getPosition();
                    this.position.set(newPosition);
                    if (this.triggerSaveState()) {
                        this.doSaveState(IndexerState.INDEXING, newPosition, this::nextSearch);
                        break block10;
                    }
                    this.nextSearch();
                }
                catch (Exception e) {
                    this.finishWithFailure(e);
                }
            }
            catch (Exception e) {
                this.finishWithSearchFailure(e);
            }
        }
    }

    private void onBulkResponse(BulkResponse response, JobPosition jobPosition) {
        ((IndexerJobStats)this.stats).markEndIndexing();
        if (!this.checkState(this.getState())) {
            return;
        }
        try {
            if (this.triggerSaveState()) {
                this.doSaveState(IndexerState.INDEXING, jobPosition, this::nextSearch);
            } else {
                this.nextSearch();
            }
        }
        catch (Exception e) {
            this.finishWithIndexingFailure(e);
        }
    }

    protected void nextSearch() {
        TimeValue executionDelay;
        this.currentMaxDocsPerSecond = this.getMaxDocsPerSecond();
        if (this.currentMaxDocsPerSecond > 0.0f && this.lastDocCount > 0L && (executionDelay = AsyncTwoPhaseIndexer.calculateThrottlingDelay(this.currentMaxDocsPerSecond, this.lastDocCount, this.lastSearchStartTimeNanos, this.getTimeNanos())).duration() > 0L) {
            logger.debug("throttling job [{}], wait for {} ({} {})", (Object)this.getJobId(), (Object)executionDelay, (Object)Float.valueOf(this.currentMaxDocsPerSecond), (Object)this.lastDocCount);
            this.scheduledNextSearch = new ScheduledRunnable(this.threadPool, executionDelay, () -> this.triggerNextSearch(executionDelay.getNanos()));
            if (this.getState().equals(IndexerState.STOPPING) || this.triggerSaveState()) {
                this.runSearchImmediately();
            }
            return;
        }
        this.triggerNextSearch(0L);
    }

    private void triggerNextSearch(long waitTimeInNanos) {
        if (!this.checkState(this.getState())) {
            return;
        }
        this.scheduledNextSearch = null;
        ((IndexerJobStats)this.stats).markStartSearch();
        this.lastSearchStartTimeNanos = this.getTimeNanos();
        this.doNextSearch(waitTimeInNanos, this.searchResponseListener);
    }

    private boolean checkState(IndexerState currentState) {
        switch (currentState) {
            case INDEXING: {
                return true;
            }
            case STOPPING: {
                logger.info("Indexer job encountered [" + String.valueOf(IndexerState.STOPPING) + "] state, halting indexer.");
                this.finishJob();
                return false;
            }
            case STOPPED: {
                return false;
            }
            case ABORTING: {
                logger.info("Requested shutdown of indexer for job [" + this.getJobId() + "]");
                this.onAbort();
                return false;
            }
        }
        logger.warn("Encountered unexpected state [" + String.valueOf(currentState) + "] while indexing");
        throw new IllegalStateException("Indexer job encountered an illegal state [" + String.valueOf(currentState) + "]");
    }

    private void reQueueThrottledSearch() {
        this.currentMaxDocsPerSecond = this.getMaxDocsPerSecond();
        ScheduledRunnable runnable = this.scheduledNextSearch;
        if (runnable != null) {
            TimeValue executionDelay = AsyncTwoPhaseIndexer.calculateThrottlingDelay(this.currentMaxDocsPerSecond, this.lastDocCount, this.lastSearchStartTimeNanos, this.getTimeNanos());
            logger.trace("[{}] rethrottling job, wait {} until next search", (Object)this.getJobId(), (Object)executionDelay);
            runnable.reschedule(executionDelay);
        }
    }

    static TimeValue calculateThrottlingDelay(float docsPerSecond, long docCount, long startTimeNanos, long now) {
        if (docsPerSecond <= 0.0f) {
            return TimeValue.ZERO;
        }
        float timeToWaitNanos = (float)docCount / docsPerSecond * (float)TimeUnit.SECONDS.toNanos(1L);
        TimeValue executionDelay = TimeValue.timeValueNanos(Math.min(MAX_THROTTLE_WAIT_TIME.getNanos(), Math.max(0L, (long)timeToWaitNanos + startTimeNanos - now)));
        if (executionDelay.compareTo(MIN_THROTTLE_WAIT_TIME) < 0) {
            return TimeValue.ZERO;
        }
        return executionDelay;
    }

    static class ScheduledRunnable {
        private final ThreadPool threadPool;
        private final Runnable command;
        private Scheduler.ScheduledCancellable scheduled;

        ScheduledRunnable(ThreadPool threadPool, TimeValue delay, Runnable command) {
            this.threadPool = threadPool;
            this.command = new RunOnce(command);
            this.scheduled = threadPool.schedule(command, delay, threadPool.generic());
        }

        public void reschedule(TimeValue delay) {
            if (this.scheduled.cancel()) {
                if (delay.duration() > 0L) {
                    this.scheduled = this.threadPool.schedule(this.command, delay, this.threadPool.generic());
                } else {
                    this.threadPool.generic().execute(this.command);
                }
            }
        }
    }
}

