/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.ReferenceDocs;
import org.elasticsearch.common.ValidationException;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.Index;

public class ShardLimitValidator {
    public static final Setting<Integer> SETTING_CLUSTER_MAX_SHARDS_PER_NODE = Setting.intSetting("cluster.max_shards_per_node", 1000, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN = Setting.intSetting("cluster.max_shards_per_node.frozen", 3000, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final String FROZEN_GROUP = "frozen";
    public static final String NORMAL_GROUP = "normal";
    static final Set<String> VALID_GROUPS = Set.of("normal", "frozen");
    public static final Setting<String> INDEX_SETTING_SHARD_LIMIT_GROUP = Setting.simpleString("index.shard_limit.group", "normal", value -> {
        if (!VALID_GROUPS.contains(value)) {
            throw new IllegalArgumentException("[" + value + "] is not a valid shard limit group");
        }
    }, Setting.Property.IndexScope, Setting.Property.PrivateIndex, Setting.Property.NotCopyableOnResize);
    protected final AtomicInteger shardLimitPerNode = new AtomicInteger();
    protected final AtomicInteger shardLimitPerNodeFrozen = new AtomicInteger();

    public ShardLimitValidator(Settings settings, ClusterService clusterService) {
        this.shardLimitPerNode.set(SETTING_CLUSTER_MAX_SHARDS_PER_NODE.get(settings));
        this.shardLimitPerNodeFrozen.set(SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN.get(settings));
        clusterService.getClusterSettings().addSettingsUpdateConsumer(SETTING_CLUSTER_MAX_SHARDS_PER_NODE, this::setShardLimitPerNode);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(SETTING_CLUSTER_MAX_SHARDS_PER_NODE_FROZEN, this::setShardLimitPerNodeFrozen);
    }

    private void setShardLimitPerNode(int newValue) {
        this.shardLimitPerNode.set(newValue);
    }

    private void setShardLimitPerNodeFrozen(int newValue) {
        this.shardLimitPerNodeFrozen.set(newValue);
    }

    public int getShardLimitPerNode() {
        return this.shardLimitPerNode.get();
    }

    public void validateShardLimit(Settings settings, DiscoveryNodes discoveryNodes, Metadata metadata) {
        int numberOfShards = IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(settings);
        int numberOfReplicas = IndexMetadata.INDEX_NUMBER_OF_REPLICAS_SETTING.get(settings);
        int shardsToCreate = numberOfShards * (1 + numberOfReplicas);
        boolean frozen = FROZEN_GROUP.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(settings));
        Result result = this.checkShardLimitOnBothGroups(!frozen ? shardsToCreate : 0, frozen ? shardsToCreate : 0, discoveryNodes, metadata);
        if (!result.canAddShards) {
            ValidationException e = new ValidationException();
            e.addValidationError(ShardLimitValidator.errorMessageFrom(result));
            throw e;
        }
    }

    public void validateShardLimit(DiscoveryNodes discoveryNodes, Metadata metadata, Index[] indicesToOpen) {
        int frozen = 0;
        int normal = 0;
        for (Index index : indicesToOpen) {
            IndexMetadata imd = metadata.index(index);
            if (!imd.getState().equals(IndexMetadata.State.CLOSE)) continue;
            int totalNewShards = imd.getNumberOfShards() * (1 + imd.getNumberOfReplicas());
            if (FROZEN_GROUP.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(imd.getSettings()))) {
                frozen += totalNewShards;
                continue;
            }
            normal += totalNewShards;
        }
        Result result = this.checkShardLimitOnBothGroups(normal, frozen, discoveryNodes, metadata);
        if (!result.canAddShards) {
            ValidationException ex = new ValidationException();
            ex.addValidationError(ShardLimitValidator.errorMessageFrom(result));
            throw ex;
        }
    }

    public void validateShardLimitOnReplicaUpdate(DiscoveryNodes discoveryNodes, Metadata metadata, Index[] indices, int replicas) {
        int frozen = 0;
        int normal = 0;
        for (Index index : indices) {
            IndexMetadata imd = metadata.index(index);
            int totalNewShards = ShardLimitValidator.getTotalNewShards(index, metadata, replicas);
            if (FROZEN_GROUP.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(imd.getSettings()))) {
                frozen += totalNewShards;
                continue;
            }
            normal += totalNewShards;
        }
        Result result = this.checkShardLimitOnBothGroups(normal, frozen, discoveryNodes, metadata);
        if (!result.canAddShards) {
            ValidationException ex = new ValidationException();
            ex.addValidationError(ShardLimitValidator.errorMessageFrom(result));
            throw ex;
        }
    }

    private static int getTotalNewShards(Index index, Metadata metadata, int updatedNumberOfReplicas) {
        IndexMetadata indexMetadata = metadata.index(index);
        int shardsInIndex = indexMetadata.getNumberOfShards();
        int oldNumberOfReplicas = indexMetadata.getNumberOfReplicas();
        int replicaIncrease = updatedNumberOfReplicas - oldNumberOfReplicas;
        return replicaIncrease * shardsInIndex;
    }

    private Result checkShardLimitOnBothGroups(int newShards, int newFrozenShards, DiscoveryNodes discoveryNodes, Metadata metadata) {
        int frozenNodeCount = ShardLimitValidator.nodeCount(discoveryNodes, ShardLimitValidator::hasFrozen);
        int normalNodeCount = ShardLimitValidator.nodeCount(discoveryNodes, ShardLimitValidator::hasNonFrozen);
        Result result = ShardLimitValidator.checkShardLimit(newShards, metadata, this.getShardLimitPerNode(), normalNodeCount, NORMAL_GROUP);
        if (!result.canAddShards()) {
            return result;
        }
        return ShardLimitValidator.checkShardLimit(newFrozenShards, metadata, this.shardLimitPerNodeFrozen.get(), frozenNodeCount, FROZEN_GROUP);
    }

    public static Result checkShardLimitForNormalNodes(int maxConfiguredShardsPerNode, int numberOfNewShards, int replicas, DiscoveryNodes discoveryNodes, Metadata metadata) {
        return ShardLimitValidator.checkShardLimit(numberOfNewShards * (1 + replicas), metadata, maxConfiguredShardsPerNode, ShardLimitValidator.nodeCount(discoveryNodes, ShardLimitValidator::hasNonFrozen), NORMAL_GROUP);
    }

    public static Result checkShardLimitForFrozenNodes(int maxConfiguredShardsPerNode, int numberOfNewShards, int replicas, DiscoveryNodes discoveryNodes, Metadata metadata) {
        return ShardLimitValidator.checkShardLimit(numberOfNewShards * (1 + replicas), metadata, maxConfiguredShardsPerNode, ShardLimitValidator.nodeCount(discoveryNodes, ShardLimitValidator::hasFrozen), FROZEN_GROUP);
    }

    private static Result checkShardLimit(int newShards, Metadata metadata, int maxConfiguredShardsPerNode, int nodeCount, String group) {
        int maxShardsInCluster = maxConfiguredShardsPerNode * nodeCount;
        int currentOpenShards = metadata.getTotalOpenIndexShards();
        if (nodeCount == 0 || newShards <= 0) {
            return new Result(true, Optional.empty(), newShards, maxShardsInCluster, group);
        }
        if (currentOpenShards + newShards > maxShardsInCluster) {
            Predicate<IndexMetadata> indexMetadataPredicate = imd -> imd.getState().equals(IndexMetadata.State.OPEN) && group.equals(INDEX_SETTING_SHARD_LIMIT_GROUP.get(imd.getSettings()));
            long currentFilteredShards = metadata.indices().values().stream().filter(indexMetadataPredicate).mapToInt(IndexMetadata::getTotalNumberOfShards).sum();
            if (currentFilteredShards + (long)newShards > (long)maxShardsInCluster) {
                return new Result(false, Optional.of(currentFilteredShards), newShards, maxShardsInCluster, group);
            }
        }
        return new Result(true, Optional.empty(), newShards, maxShardsInCluster, group);
    }

    private static int nodeCount(DiscoveryNodes discoveryNodes, Predicate<DiscoveryNode> nodePredicate) {
        return (int)discoveryNodes.getDataNodes().values().stream().filter(nodePredicate).count();
    }

    private static boolean hasFrozen(DiscoveryNode node) {
        return node.getRoles().contains(DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE);
    }

    private static boolean hasNonFrozen(DiscoveryNode node) {
        return node.getRoles().stream().anyMatch(r -> r.canContainData() && r != DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE);
    }

    static String errorMessageFrom(Result result) {
        return "this action would add [" + result.totalShardsToAdd + "] shards, but this cluster currently has [" + String.valueOf(result.currentUsedShards.get()) + "]/[" + result.maxShardsInCluster + "] maximum " + result.group + " shards open; for more information, see " + String.valueOf((Object)ReferenceDocs.MAX_SHARDS_PER_NODE);
    }

    public record Result(boolean canAddShards, Optional<Long> currentUsedShards, int totalShardsToAdd, int maxShardsInCluster, String group) {
    }
}

