/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.health;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.health.HealthIndicatorDetails;
import org.elasticsearch.health.HealthIndicatorResult;
import org.elasticsearch.health.HealthIndicatorService;
import org.elasticsearch.health.HealthStatus;
import org.elasticsearch.health.SimpleHealthIndicatorDetails;
import org.elasticsearch.health.node.FetchHealthInfoCacheAction;
import org.elasticsearch.health.node.HealthInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class HealthService {
    static final String UNKNOWN_RESULT_SUMMARY_PREFLIGHT_FAILED = "Could not determine health status. Check details on critical issues preventing the health status from reporting.";
    public static final String HEALTH_API_ID_PREFIX = "elasticsearch:health:";
    private static final String REASON = "reasons";
    private final List<HealthIndicatorService> preflightHealthIndicatorServices;
    private final List<HealthIndicatorService> healthIndicatorServices;
    private final ThreadPool threadPool;

    public HealthService(List<HealthIndicatorService> healthIndicatorServices, ThreadPool threadPool) {
        this.preflightHealthIndicatorServices = healthIndicatorServices.stream().filter(HealthIndicatorService::isPreflight).toList();
        this.healthIndicatorServices = healthIndicatorServices.stream().filter(indicator -> !indicator.isPreflight()).toList();
        this.threadPool = threadPool;
    }

    public void getHealth(Client client, final @Nullable String indicatorName, final boolean verbose, final int maxAffectedResourcesCount, final ActionListener<List<HealthIndicatorResult>> listener) {
        if (maxAffectedResourcesCount < 0) {
            throw new IllegalArgumentException("The max number of resources must be a positive integer");
        }
        List<HealthIndicatorResult> preflightResults = this.preflightHealthIndicatorServices.stream().map(service -> service.calculate(verbose, maxAffectedResourcesCount, HealthInfo.EMPTY_HEALTH_INFO)).toList();
        boolean clusterHealthIsObtainable = preflightResults.isEmpty() || preflightResults.stream().map(HealthIndicatorResult::status).allMatch(Predicate.isEqual(HealthStatus.GREEN));
        final Stream<HealthIndicatorService> filteredIndicators = this.healthIndicatorServices.stream().filter(service -> indicatorName == null || service.name().equals(indicatorName));
        final Stream<HealthIndicatorResult> filteredPreflightResults = preflightResults.stream().filter(result -> indicatorName == null || result.name().equals(indicatorName));
        if (clusterHealthIsObtainable) {
            client.execute(FetchHealthInfoCacheAction.INSTANCE, new FetchHealthInfoCacheAction.Request(), new ActionListener<FetchHealthInfoCacheAction.Response>(){

                @Override
                public void onResponse(FetchHealthInfoCacheAction.Response response) {
                    HealthInfo healthInfo = response.getHealthInfo();
                    ActionRunnable<List<HealthIndicatorResult>> calculateFilteredIndicatorsRunnable = this.calculateFilteredIndicatorsRunnable(indicatorName, healthInfo, verbose, listener);
                    try {
                        HealthService.this.threadPool.executor("management").submit(calculateFilteredIndicatorsRunnable);
                    }
                    catch (EsRejectedExecutionException e) {
                        calculateFilteredIndicatorsRunnable.onRejection(e);
                    }
                }

                @Override
                public void onFailure(Exception e) {
                    ActionRunnable<List<HealthIndicatorResult>> calculateFilteredIndicatorsRunnable = this.calculateFilteredIndicatorsRunnable(indicatorName, HealthInfo.EMPTY_HEALTH_INFO, verbose, listener);
                    try {
                        HealthService.this.threadPool.executor("management").submit(calculateFilteredIndicatorsRunnable);
                    }
                    catch (EsRejectedExecutionException esRejectedExecutionException) {
                        calculateFilteredIndicatorsRunnable.onRejection(esRejectedExecutionException);
                    }
                }

                private ActionRunnable<List<HealthIndicatorResult>> calculateFilteredIndicatorsRunnable(String indicatorName2, HealthInfo healthInfo, boolean explain, ActionListener<List<HealthIndicatorResult>> listener2) {
                    return ActionRunnable.wrap(listener2, (ActionListener<T> l) -> {
                        List<HealthIndicatorResult> results = Stream.concat(filteredPreflightResults, filteredIndicators.map((? super T service) -> service.calculate(explain, maxAffectedResourcesCount, healthInfo))).toList();
                        HealthService.validateResultsAndNotifyListener(indicatorName2, results, l);
                    });
                }
            });
        } else {
            HealthIndicatorDetails unknownDetails = HealthService.healthUnknownReason(preflightResults, verbose);
            Stream<HealthIndicatorResult> filteredIndicatorResults = filteredIndicators.map(service -> HealthService.generateUnknownResult(service, UNKNOWN_RESULT_SUMMARY_PREFLIGHT_FAILED, unknownDetails));
            HealthService.validateResultsAndNotifyListener(indicatorName, Stream.concat(filteredPreflightResults, filteredIndicatorResults).toList(), listener);
        }
    }

    private static void validateResultsAndNotifyListener(@Nullable String indicatorName, List<HealthIndicatorResult> results, ActionListener<List<HealthIndicatorResult>> listener) {
        assert (HealthService.findDuplicatesByName(results).isEmpty()) : String.format(Locale.ROOT, "Found multiple indicators with the same name: %s", HealthService.findDuplicatesByName(results));
        if (results.isEmpty() && indicatorName != null) {
            String errorMessage = String.format(Locale.ROOT, "Did not find indicator %s", indicatorName);
            listener.onFailure(new ResourceNotFoundException(errorMessage, new Object[0]));
        } else {
            listener.onResponse(results);
        }
    }

    private static HealthIndicatorDetails healthUnknownReason(List<HealthIndicatorResult> preflightResults, boolean computeDetails) {
        HealthIndicatorDetails unknownDetails;
        assert (!preflightResults.isEmpty()) : "Requires at least one non-GREEN preflight result";
        if (computeDetails) {
            Map<String, String> clusterUnstableReasons = preflightResults.stream().filter(result -> !HealthStatus.GREEN.equals(result.status())).collect(Collectors.toMap(HealthIndicatorResult::name, result -> result.status().xContentValue()));
            assert (!clusterUnstableReasons.isEmpty()) : "Requires at least one non-GREEN preflight result";
            unknownDetails = new SimpleHealthIndicatorDetails(Map.of(REASON, clusterUnstableReasons));
        } else {
            unknownDetails = HealthIndicatorDetails.EMPTY;
        }
        return unknownDetails;
    }

    private static HealthIndicatorResult generateUnknownResult(HealthIndicatorService indicatorService, String summary, HealthIndicatorDetails details) {
        return indicatorService.createIndicator(HealthStatus.UNKNOWN, summary, details, Collections.emptyList(), Collections.emptyList());
    }

    private static Set<String> findDuplicatesByName(List<HealthIndicatorResult> indicators) {
        HashSet items = new HashSet();
        return indicators.stream().map(HealthIndicatorResult::name).filter(name -> !items.add(name)).collect(Collectors.toSet());
    }
}

