"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEntityInstancesQuery = getEntityInstancesQuery;
var _esQuery = require("@kbn/es-query");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fromCommand = ({
  source
}) => {
  let query = `FROM ${source.index_patterns.join(', ')}`;
  const esMetadataFields = source.metadata_fields.filter(field => ['_index', '_id'].includes(field));
  if (esMetadataFields.length) {
    query += ` METADATA ${esMetadataFields.join(', ')}`;
  }
  return query;
};
const dslFilter = ({
  source,
  start,
  end
}) => {
  const filters = [...source.filters, ...source.identity_fields.map(field => `${field}: *`)];
  if (source.timestamp_field) {
    filters.push(`${source.timestamp_field} >= "${start}" AND ${source.timestamp_field} <= "${end}"`);
  }
  const kuery = filters.map(filter => '(' + filter + ')').join(' AND ');
  return (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery));
};
const statsCommand = ({
  source
}) => {
  const aggs = source.metadata_fields.filter(field => !source.identity_fields.some(idField => idField === field)).map(field => `${field} = VALUES(${(0, _utils.asKeyword)(field)})`);
  if (source.timestamp_field) {
    aggs.push(`entity.last_seen_timestamp = MAX(${source.timestamp_field})`);
  }
  if (source.display_name) {
    // ideally we want the latest value but there's no command yet
    // so we use MAX for now
    aggs.push(`${source.display_name} = MAX(${(0, _utils.asKeyword)(source.display_name)})`);
  }
  return `STATS ${aggs.join(', ')} BY ${source.identity_fields.map(_utils.asKeyword).join(', ')}`;
};
const renameCommand = ({
  source
}) => {
  const operations = source.identity_fields.map(field => `\`${(0, _utils.asKeyword)(field)}\` AS ${field}`);
  return `RENAME ${operations.join(', ')}`;
};
const evalCommand = ({
  source
}) => {
  const id = source.identity_fields.length === 1 ? source.identity_fields[0] : `CONCAT(${source.identity_fields.join(', ":", ')})`;
  const displayName = source.display_name ? `COALESCE(${source.display_name}, entity.id)` : 'entity.id';
  return `EVAL ${[`entity.type = "${source.type_id}"`, `entity.id = ${id}`, `entity.display_name = ${displayName}`].join(', ')}`;
};
const sortCommand = ({
  source,
  sort
}) => {
  if (sort) {
    return `SORT ${sort.field} ${sort.direction}`;
  }
  if (source.timestamp_field) {
    return `SORT entity.last_seen_timestamp DESC`;
  }
  return `SORT entity.id ASC`;
};
function getEntityInstancesQuery({
  source,
  limit,
  start,
  end,
  sort
}) {
  const commands = [fromCommand({
    source
  }), statsCommand({
    source
  }), renameCommand({
    source
  }), evalCommand({
    source
  }), sortCommand({
    source,
    sort
  }), `LIMIT ${limit}`];
  const filter = dslFilter({
    source,
    start,
    end
  });
  return {
    query: commands.join(' | '),
    filter
  };
}