"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSavedQueryRouteHandlerContext = registerSavedQueryRouteHandlerContext;
var _boom = require("@hapi/boom");
var _esQuery = require("@kbn/es-query");
var _lodash = require("lodash");
var _common = require("../../common");
var _persistable_state = require("../../common/query/filters/persistable_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function injectReferences({
  id,
  attributes: internalAttributes,
  namespaces,
  references
}) {
  const attributes = (0, _lodash.omit)(internalAttributes, 'titleKeyword', 'filters', 'timefilter');

  // filters or timefilter can be null if previously removed in an update,
  // which isn't valid for the client model, so we conditionally add them
  if (internalAttributes.filters) {
    attributes.filters = (0, _persistable_state.inject)(internalAttributes.filters, references);
  }
  if (internalAttributes.timefilter) {
    attributes.timefilter = internalAttributes.timefilter;
  }
  const {
    query
  } = attributes;
  if ((0, _esQuery.isOfQueryType)(query) && typeof query.query === 'string') {
    try {
      const parsed = JSON.parse(query.query);
      query.query = parsed instanceof Object ? parsed : query.query;
    } catch (e) {
      // Just keep it as a string
    }
  }
  return {
    id,
    attributes,
    namespaces
  };
}
function extractReferences({
  title,
  description,
  query,
  filters,
  timefilter
}) {
  const {
    state: extractedFilters,
    references
  } = filters ? (0, _persistable_state.extract)(filters) : {
    state: undefined,
    references: []
  };
  const isOfQueryTypeQuery = (0, _esQuery.isOfQueryType)(query);
  let queryString = '';
  if (isOfQueryTypeQuery) {
    if (typeof query.query === 'string') {
      queryString = query.query;
    } else {
      queryString = JSON.stringify(query.query);
    }
  }
  const attributes = {
    title: title.trim(),
    titleKeyword: title.trim(),
    description: description.trim(),
    query: {
      ...query,
      ...(queryString && {
        query: queryString
      })
    },
    // Pass null instead of undefined for filters and timefilter
    // to ensure they are removed from the saved object on update
    // since the saved objects client ignores undefined values
    filters: extractedFilters !== null && extractedFilters !== void 0 ? extractedFilters : null,
    timefilter: timefilter !== null && timefilter !== void 0 ? timefilter : null
  };
  return {
    attributes,
    references
  };
}
function verifySavedQuery({
  title,
  query,
  filters = []
}) {
  if (!(0, _common.isQuery)(query)) {
    throw (0, _boom.badRequest)(`Invalid query: ${JSON.stringify(query, null, 2)}`);
  }
  if (!(0, _esQuery.isFilters)(filters)) {
    throw (0, _boom.badRequest)(`Invalid filters: ${JSON.stringify(filters, null, 2)}`);
  }
  if (!title.trim().length) {
    throw (0, _boom.badRequest)('Cannot create query without a title');
  }
}
async function registerSavedQueryRouteHandlerContext(context) {
  const soClient = (await context.core).savedObjects.client;
  const isDuplicateTitle = async ({
    title,
    id
  }) => {
    const preparedTitle = title.trim();
    const {
      saved_objects: savedQueries
    } = await soClient.find({
      type: 'query',
      page: 1,
      perPage: 1,
      filter: `query.attributes.titleKeyword:"${(0, _esQuery.escapeQuotes)(preparedTitle)}"`
    });
    const existingQuery = savedQueries[0];
    return Boolean(existingQuery && existingQuery.attributes.titleKeyword === preparedTitle && (!id || existingQuery.id !== id));
  };
  const validateSavedQueryTitle = async (title, id) => {
    if (await isDuplicateTitle({
      title,
      id
    })) {
      throw (0, _boom.badRequest)(`Query with title "${title.trim()}" already exists`);
    }
  };
  const createSavedQuery = async attrs => {
    verifySavedQuery(attrs);
    await validateSavedQueryTitle(attrs.title);
    const {
      attributes,
      references
    } = extractReferences(attrs);
    const savedObject = await soClient.create('query', attributes, {
      references
    });

    // TODO: Handle properly
    if (savedObject.error) throw (0, _boom.internal)(savedObject.error.message);
    return injectReferences(savedObject);
  };
  const updateSavedQuery = async (id, attrs) => {
    verifySavedQuery(attrs);
    await validateSavedQueryTitle(attrs.title, id);
    const {
      attributes,
      references
    } = extractReferences(attrs);
    const savedObject = await soClient.update('query', id, attributes, {
      references
    });

    // TODO: Handle properly
    if (savedObject.error) throw (0, _boom.internal)(savedObject.error.message);
    return injectReferences({
      id,
      attributes,
      references
    });
  };
  const getSavedQuery = async id => {
    const {
      saved_object: savedObject,
      outcome
    } = await soClient.resolve('query', id);
    if (outcome === 'conflict') {
      throw (0, _boom.conflict)(`Multiple saved queries found with ID: ${id} (legacy URL alias conflict)`);
    } else if (savedObject.error) {
      throw (0, _boom.internal)(savedObject.error.message);
    }
    return injectReferences(savedObject);
  };
  const getSavedQueriesCount = async () => {
    const {
      total
    } = await soClient.find({
      type: 'query',
      page: 0,
      perPage: 0
    });
    return total;
  };
  const findSavedQueries = async ({
    page = 1,
    perPage = 50,
    search = ''
  } = {}) => {
    const cleanedSearch = search.replace(/\W/g, ' ').trim();
    const preparedSearch = (0, _esQuery.escapeKuery)(cleanedSearch).split(/\s+/).join(' AND ');
    const {
      total,
      saved_objects: savedObjects
    } = await soClient.find({
      type: 'query',
      page,
      perPage,
      filter: preparedSearch.length ? `query.attributes.title:(*${preparedSearch}*)` : undefined,
      sortField: 'titleKeyword',
      sortOrder: 'asc'
    });
    const savedQueries = savedObjects.map(injectReferences);
    return {
      total,
      savedQueries
    };
  };
  const deleteSavedQuery = async id => {
    return await soClient.delete('query', id, {
      force: true
    });
  };
  return {
    isDuplicateTitle,
    create: createSavedQuery,
    update: updateSavedQuery,
    get: getSavedQuery,
    count: getSavedQueriesCount,
    find: findSavedQueries,
    delete: deleteSavedQuery
  };
}