"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkForDuplicateDashboardTitle = checkForDuplicateDashboardTitle;
var _telemetry_constants = require("../../../utils/telemetry_constants");
var _extract_title_and_count = require("../../../dashboard_container/embeddable/api/lib/extract_title_and_count");
var _kibana_services = require("../../kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * check for an existing dashboard with the same title in ES
 * returns Promise<true> when there is no duplicate, or runs the provided onTitleDuplicate
 * function when the title already exists
 */
async function checkForDuplicateDashboardTitle({
  title,
  copyOnSave,
  lastSavedTitle,
  onTitleDuplicate,
  isTitleDuplicateConfirmed
}) {
  // Don't check if the title is an empty string
  if (!title) {
    return true;
  }

  // Don't check for duplicates if user has already confirmed save with duplicate title
  if (isTitleDuplicateConfirmed) {
    return true;
  }

  // Don't check if the user isn't updating the title, otherwise that would become very annoying to have
  // to confirm the save every time, except when copyOnSave is true, then we do want to check.
  if (title === lastSavedTitle && !copyOnSave) {
    return true;
  }
  const [baseDashboardName] = (0, _extract_title_and_count.extractTitleAndCount)(title);
  const {
    hits
  } = await _kibana_services.contentManagementService.client.search({
    contentTypeId: _telemetry_constants.DASHBOARD_CONTENT_ID,
    query: {
      text: `${baseDashboardName}*`,
      limit: 20
    },
    options: {
      onlyTitle: true
    }
  });
  const duplicate = Boolean(hits.find(hit => hit.attributes.title.toLowerCase() === title.toLowerCase()));
  if (!duplicate) {
    return true;
  }
  const [largestDuplicationId] = hits.map(hit => (0, _extract_title_and_count.extractTitleAndCount)(hit.attributes.title)[1]).sort((a, b) => b - a);
  const speculativeCollisionFreeTitle = `${baseDashboardName} (${largestDuplicationId + 1})`;
  onTitleDuplicate === null || onTitleDuplicate === void 0 ? void 0 : onTitleDuplicate(speculativeCollisionFreeTitle);
  return false;
}