"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EsEventStreamInitializer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pRetry = _interopRequireDefault(require("p-retry"));
var _index_template = require("./index_template");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
var _retry = /*#__PURE__*/new WeakMap();
class EsEventStreamInitializer {
  constructor(deps) {
    /**
     * Calls a function; retries calling it multiple times via p-retry, if it fails.
     * Should retry on 2s, 4s, 8s, 16s.
     *
     * See: https://github.com/tim-kos/node-retry#retryoperationoptions
     *
     * @param fn Function to retry, if it fails.
     */
    _classPrivateFieldInitSpec(this, _retry, async (fn, fnName) => {
      this.deps.logger.debug(`Event Stream initialization operation: ${fnName}`);
      return await (0, _pRetry.default)(fn, {
        minTimeout: 1000,
        maxTimeout: 1000 * 60 * 3,
        retries: 4,
        factor: 2,
        randomize: true,
        onFailedAttempt: err => {
          const message = `Event Stream initialization operation failed and will be retried: ${fnName};` + `${err.retriesLeft} more times; error: ${err.message}`;
          this.deps.logger.warn(message);
        }
      });
    });
    (0, _defineProperty2.default)(this, "createIndexTemplateIfNotExists", async () => {
      const exists = await this.indexTemplateExists();
      if (exists) return false;
      return await this.createIndexTemplate();
    });
    (0, _defineProperty2.default)(this, "createDataStream", async () => {
      const esClient = await this.deps.esClient;
      const name = this.deps.names.dataStream;
      try {
        await esClient.indices.createDataStream({
          name
        });
      } catch (error) {
        var _body, _body$error;
        const alreadyExists = (error === null || error === void 0 ? void 0 : (_body = error.body) === null || _body === void 0 ? void 0 : (_body$error = _body.error) === null || _body$error === void 0 ? void 0 : _body$error.type) === 'resource_already_exists_exception';
        if (alreadyExists) return;
        throw error;
      }
    });
    this.deps = deps;
  }
  async initialize() {
    const createdIndexTemplate = await _classPrivateFieldGet(_retry, this).call(this, this.createIndexTemplateIfNotExists, 'createIndexTemplateIfNotExists');
    if (createdIndexTemplate) {
      await _classPrivateFieldGet(_retry, this).call(this, this.createDataStream, 'createDataStream');
    }
  }
  async indexTemplateExists() {
    try {
      const esClient = await this.deps.esClient;
      const name = this.deps.names.indexTemplate;
      const exists = await esClient.indices.existsIndexTemplate({
        name
      });
      return !!exists;
    } catch (err) {
      throw new Error(`error checking existence of index template: ${err.message}`);
    }
  }
  async createIndexTemplate() {
    try {
      const esClient = await this.deps.esClient;
      const {
        indexTemplate,
        indexPattern
      } = this.deps.names;
      const request = (0, _index_template.newIndexTemplateRequest)({
        name: indexTemplate,
        indexPatterns: [indexPattern],
        kibanaVersion: this.deps.kibanaVersion
      });
      await esClient.indices.putIndexTemplate(request);
      return true;
    } catch (err) {
      // The error message doesn't have a type attribute we can look to guarantee it's due
      // to the template already existing (only long message) so we'll check ourselves to see
      // if the template now exists. This scenario would happen if you startup multiple Kibana
      // instances at the same time.
      const exists = await this.indexTemplateExists();
      if (exists) return false;
      const error = new Error(`error creating index template: ${err.message}`);
      Object.assign(error, {
        wrapped: err
      });
      throw error;
    }
  }
}
exports.EsEventStreamInitializer = EsEventStreamInitializer;